/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import java.io.IOException;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.reflect.ClassPath;
import com.google.common.reflect.ClassPath.ClassInfo;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.environment.ApplicationInfoProvider;
import com.sap.cds.services.environment.PropertiesProvider;
import com.sap.cds.services.environment.ServiceBindingProvider;
import com.sap.cds.services.messages.LocalizedMessageProvider;
import com.sap.cds.services.runtime.AuthenticationInfoProvider;
import com.sap.cds.services.runtime.CdsModelProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.runtime.ExtendedServiceLoader;
import com.sap.cds.services.runtime.FeatureTogglesInfoProvider;
import com.sap.cds.services.runtime.ParameterInfoProvider;
import com.sap.cds.services.runtime.UserInfoProvider;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

public class CdsRuntimeConfigurerImpl implements CdsRuntimeConfigurer {

	private static final Logger logger = LoggerFactory.getLogger(CdsRuntimeConfigurerImpl.class);

	private static enum ConfigurationPhase {
		CONFIGURATIONS,
		ENVIRONMENT,
		MODEL,
		SERVICES,
		EVENTHANDLERS,
		PROVIDERS,
		COMPLETE
	}

	private final CdsRuntimeImpl runtime;
	private final List<CdsRuntimeConfiguration> configurations = new ArrayList<>();
	private ConfigurationPhase current = ConfigurationPhase.CONFIGURATIONS;

	public CdsRuntimeConfigurerImpl(PropertiesProvider propertiesProvider) {
		runtime = new CdsRuntimeImpl(propertiesProvider);
		ExtendedServiceLoader.loadAll(CdsRuntimeConfiguration.class).forEachRemaining(this::configuration);
	}

	@Override
	public CdsRuntime getCdsRuntime() {
		return runtime;
	}

	@Override
	public CdsRuntimeConfigurer configuration(CdsRuntimeConfiguration configuration) {
		checkPhase(ConfigurationPhase.CONFIGURATIONS);
		configurations.add(configuration);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer environmentConfigurations() {
		checkPhase(ConfigurationPhase.ENVIRONMENT);
		sortConfigurations();
		configurations.forEach(c -> c.environment(this));
		return this;
	}

	@Override
	public CdsRuntimeConfigurer environment(ServiceBindingProvider provider) {
		checkPhase(ConfigurationPhase.ENVIRONMENT);
		provider.setPrevious(runtime.getServiceBindingProvider());
		runtime.setServiceBindingProvider(provider);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer environment(ApplicationInfoProvider provider) {
		checkPhase(ConfigurationPhase.ENVIRONMENT);
		provider.setPrevious(runtime.getApplicationInfoProvider());
		runtime.setApplicationInfoProvider(provider);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer cdsModel() {
		checkPhase(ConfigurationPhase.MODEL);
		try {
			return cdsModel(runtime.getEnvironment().getCdsProperties().getModel().getCsnPath());
		} catch (ServiceException e) {
			if(e.getErrorStatus().equals(CdsErrorStatuses.INVALID_CSN)) {
				logger.debug("Failed to load default CDS model", e);
				logger.warn("Initialized empty CDS model, because default CDS model could not be loaded");
				return cdsModel((String) null); // empty model
			}
			throw e;
		}
	}

	@Override
	public CdsRuntimeConfigurer cdsModel(String csnPath) {
		checkPhase(ConfigurationPhase.MODEL);
		runtime.setCdsModel(csnPath);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer cdsModel(CdsModel model) {
		checkPhase(ConfigurationPhase.MODEL);
		runtime.setCdsModel(model);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer serviceConfigurations() {
		checkPhase(ConfigurationPhase.SERVICES);
		sortConfigurations();
		configurations.forEach(c -> c.services(this));
		return this;
	}

	@Override
	public CdsRuntimeConfigurer service(Service service) {
		checkPhase(ConfigurationPhase.SERVICES);
		runtime.registerService(service);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer eventHandlerConfigurations() {
		checkPhase(ConfigurationPhase.EVENTHANDLERS);
		sortConfigurations();
		configurations.forEach(c -> c.eventHandlers(this));
		return this;
	}

	@Override
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public CdsRuntimeConfigurer eventHandler(Object handler) {
		checkPhase(ConfigurationPhase.EVENTHANDLERS);
		return eventHandler((Class) handler.getClass(), (Supplier) () -> handler);
	}

	@Override
	public <T> CdsRuntimeConfigurer eventHandler(Class<T> handlerClass, Supplier<T> handlerFactory) {
		checkPhase(ConfigurationPhase.EVENTHANDLERS);
		runtime.registerEventHandler(handlerClass, handlerFactory);
		return this;
	}

	@Override
	@SuppressWarnings({"unchecked", "rawtypes"})
	public CdsRuntimeConfigurer packageScan(String packageName) throws IOException {
		checkPhase(ConfigurationPhase.EVENTHANDLERS);
		if(StringUtils.isEmpty(packageName)) {
			return this;
		}

		MethodHandles.Lookup lookup = MethodHandles.lookup();
		MethodType mt = MethodType.methodType(void.class);

		ClassLoader classloader = Thread.currentThread().getContextClassLoader();
		for (ClassInfo classInfo : ClassPath.from(classloader).getAllClasses()) {

			if (!classInfo.getName().startsWith(packageName)) {
				continue;
			}

			// need c'tor without arguments for supplier. Note that this supplier creates a
			// new handler instance on each request.
			try {
				Class<?> testClass = Class.forName(classInfo.getName());
				MethodHandle defaultCtor = lookup.findConstructor(testClass, mt);
				Supplier<?> supplier = () -> {
					try {
						return defaultCtor.invoke();
					} catch (Throwable e) { // NOSONAR
						throw new ErrorStatusException(CdsErrorStatuses.HANDLER_FAILED, testClass.getName(), e);
					}
				};
				logger.info("Found potential Event Handler class {}", testClass.getName());
				eventHandler((Class) testClass, (Supplier) supplier);
			} catch(ClassNotFoundException | IllegalAccessException | NoSuchMethodException e) { // NOSONAR
				// ignore
			}
		}
		return this;
	}

	@Override
	public CdsRuntimeConfigurer providerConfigurations() {
		checkPhase(ConfigurationPhase.PROVIDERS);
		sortConfigurations();
		configurations.forEach(c -> c.providers(this));
		return this;
	}

	@Override
	public CdsRuntimeConfigurer provider(CdsModelProvider provider) {
		checkPhase(ConfigurationPhase.PROVIDERS);
		provider.setPrevious(runtime.getCdsModelProvider());
		runtime.setCdsModelProvider(provider);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer provider(AuthenticationInfoProvider provider) {
		checkPhase(ConfigurationPhase.PROVIDERS);
		provider.setPrevious(runtime.getAuthenticationInfoProvider());
		runtime.setAuthenticationInfoProvider(provider);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer provider(UserInfoProvider provider) {
		checkPhase(ConfigurationPhase.PROVIDERS);
		provider.setPrevious(runtime.getUserInfoProvider());
		runtime.setUserInfoProvider(provider);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer provider(ParameterInfoProvider provider) {
		checkPhase(ConfigurationPhase.PROVIDERS);
		provider.setPrevious(runtime.getParameterInfoProvider());
		runtime.setParameterInfoProvider(provider);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer provider(LocalizedMessageProvider provider) {
		checkPhase(ConfigurationPhase.PROVIDERS);
		provider.setPrevious(runtime.getLocalizedMessageProvider());
		runtime.setLocalizedMessageProvider(provider);
		return this;
	}

	@Override
	public CdsRuntimeConfigurer provider(FeatureTogglesInfoProvider provider) {
		checkPhase(ConfigurationPhase.PROVIDERS);
		provider.setPrevious(runtime.getFeatureToggleProvider());
		runtime.setFeatureToggleProvider(provider);
		return this;
	}

	@Override
	public CdsRuntime complete() {
		this.current = ConfigurationPhase.COMPLETE;
		return runtime;
	}

	private void checkPhase(ConfigurationPhase phase) {
		if(current == ConfigurationPhase.COMPLETE) {
			throw new ErrorStatusException(CdsErrorStatuses.CONFIGURER_COMPLETED);
		}

		if(current.ordinal() > phase.ordinal()) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_CONFIGURATION_PHASE, phase.name(), current.name());
		}

		this.current = phase;
	}

	private void sortConfigurations() {
		Collections.sort(configurations, (c1, c2) -> Integer.compare(c1.order(), c2.order()));
	}

}
