/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.impl.changeset.ChangeSetContextImpl;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.ChangeSetContextRunner;

public class ChangeSetContextRunnerImpl implements ChangeSetContextRunner {

	private final static Logger logger = LoggerFactory.getLogger(CdsRuntimeImpl.class);

	private final CdsRuntime runtime;

	public ChangeSetContextRunnerImpl(CdsRuntime runtime) {
		this.runtime = Objects.requireNonNull(runtime, "runtime must not be null");
	}

	@Override
	public <T> T run(Function<ChangeSetContext, T> changeSetHandler) {
		if(!RequestContext.isActive()) {
			return runtime.requestContext().run(requestContext -> {
				return open(changeSetHandler);
			});
		}
		return open(changeSetHandler);

	}

	@Override
	public void run(Consumer<ChangeSetContext> changeSetHandler) {
		run((changeSetContext) -> {
			changeSetHandler.accept(changeSetContext);
			return Void.TYPE;
		});
	}

	private <T> T open(Function<ChangeSetContext, T> handler) {
		try (ChangeSetContextImpl changeSetContext = ChangeSetContextImpl.open()) {
			try {
				return handler.apply(changeSetContext);
			} catch (Exception e) { // NOSONAR
				// cancel in case of any exception / reject
				changeSetContext.markForCancel();
				logger.info("Exception marked the ChangeSet {} as cancelled: {}", changeSetContext.getId(), e.getMessage());
				throw e;
			}
		}
	}

}
