/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.utils;

import java.util.List;
import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.impl.EventContextDelegator;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.impl.EventContextSPI;
import com.sap.cds.services.impl.ServiceSPI;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.SessionContextUtils;
import com.sap.cds.util.PathExpressionResolver;

/**
 * Utility class for CDS Services
 */
public class CdsServiceUtils {

	/**
	 * Returns the default persistence service.
	 * @param context the current context
	 * @return the default persistence service
	 */
	public static PersistenceService getDefaultPersistenceService(EventContext context) {
		return context.getServiceCatalog().getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
	}

	/**
	 * Returns the Service casted to the {@link ServiceSPI}
	 * @param service the {@link Service} to be casted
	 * @return the Service casted to the {@link ServiceSPI}, or null
	 */
	public static ServiceSPI getServiceSPI(Service service) {
		if(service instanceof ServiceSPI) {
			return (ServiceSPI) service;
		} else if (service instanceof ServiceDelegator) {
			return getServiceSPI(((ServiceDelegator) service).getDelegatedService());
		}
		return null;
	}

	/**
	 * Returns the list of entity maps present in the {@link EventContext}.
	 * This method only works for {@link CqnService#EVENT_CREATE}, {@link CqnService#EVENT_UPDATE} and {@link CqnService#EVENT_UPSERT}.
	 * It inspects the CQN of these events and returns the entity data from it.
	 *
	 * @param context the {@link EventContext}
	 * @return the list of entity maps available in the CQN of the {@link EventContext}
	 */
	public static List<Map<String, Object>> getEntities(EventContext context) {
		switch(context.getEvent()) {
		case CqnService.EVENT_CREATE:
			return context.as(CdsCreateEventContext.class).getCqn().entries();
		case CqnService.EVENT_UPDATE:
			return context.as(CdsUpdateEventContext.class).getCqn().entries();
		case CqnService.EVENT_UPSERT:
			return context.as(CdsUpsertEventContext.class).getCqn().entries();
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EVENT, context.getEvent());
		}
	}

	/**
	 * Returns the list of entity maps present in the {@link EventContext}.
	 * This method only works for {@link CqnService#EVENT_CREATE}, {@link CqnService#EVENT_UPDATE} and {@link CqnService#EVENT_UPSERT}.
	 * It inspects the CQN of these events and returns the entity data from it.
	 * In contrast to {link {@link #getEntities(EventContext)} the data also contains
	 * the resolved foreign keys from parent entities in path expressions.
	 *
	 * @param context the {@link EventContext}
	 * @return the list of entity maps available in the CQN of the {@link EventContext}
	 */
	public static List<Map<String, Object>> getEntitiesResolved(EventContext context) {
		switch(context.getEvent()) {
		case CqnService.EVENT_CREATE:
			return PathExpressionResolver.resolvePath(context.getModel(),
					context.as(CdsCreateEventContext.class).getCqn(), SessionContextUtils.toSessionContext(context)).entries();
		case CqnService.EVENT_UPDATE:
			return PathExpressionResolver.resolvePath(context.getModel(),
					context.as(CdsUpdateEventContext.class).getCqn()).entries();
		case CqnService.EVENT_UPSERT:
			return PathExpressionResolver.resolvePath(context.getModel(),
					context.as(CdsUpsertEventContext.class).getCqn(), SessionContextUtils.toSessionContext(context)).entries();
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EVENT, context.getEvent());
		}
	}

	/**
	 * Returns the {@link Result} present in the {@link EventContext}.
	 * This method only works for the default CRUD events.
	 *
	 * @param context the {@link EventContext}
	 * @return the {@link Result} present in the {@link EventContext}
	 */
	public static Result getResult(EventContext context) {
		switch(context.getEvent()) {
		case CqnService.EVENT_READ:
			return context.as(CdsReadEventContext.class).getResult();
		case CqnService.EVENT_CREATE:
			return context.as(CdsCreateEventContext.class).getResult();
		case CqnService.EVENT_UPDATE:
			return context.as(CdsUpdateEventContext.class).getResult();
		case CqnService.EVENT_UPSERT:
			return context.as(CdsUpsertEventContext.class).getResult();
		case CqnService.EVENT_DELETE:
			return context.as(CdsDeleteEventContext.class).getResult();
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNEXPECTED_EVENT, context.getEvent());
		}
	}

	/**
	 * Returns the EventContext casted to the {@link EventContextSPI}
	 * @param eventContext the {@link EventContext} to be casted
	 * @return the Service casted to the {@link EventContextSPI}, or null
	 */
	public static EventContextSPI getEventContextSPI(EventContext eventContext) {
		if(eventContext instanceof EventContextSPI) {
			return (EventContextSPI) eventContext;
		} else if (eventContext instanceof EventContextDelegator) {
			return getEventContextSPI(((EventContextDelegator) eventContext).getDelegatedEventContext());
		}
		return null;
	}
}
