/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.sap.cds.impl.parser.builder.ExpressionBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.RefBuilder;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnComparisonPredicate.Operator;
import com.sap.cds.ql.cqn.CqnConnectivePredicate;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnNegation;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnToken;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.impl.LeanModifier;
import com.sap.cds.ql.impl.Xpr;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.utils.DraftUtils;

/**
 * Helper class to adapt cqn queries for draft entities.
 */
public class CqnAdapter {

	public static final String DRAFT_SUFFIX = "_drafts";

	/**
	 * Returns the corresponding draft entity name. If the name is already
	 * that of a draft entity it is not changed.
	 * @param entity the entity
	 * @return the draft entity name
	 */
	public static String getDraftsEntity(String entity) {
		if (entity.endsWith(DRAFT_SUFFIX)) {
			return entity;
		}
		return entity + DRAFT_SUFFIX;
	}

	/**
	 * Changes references to IsActiveEntity to {@code true}.
	 * @param delete the delete statement
	 * @param context the {@link EventContext}
	 * @return a statement were all references to IsActiveEntity are changed to {@code true}
	 */
	public static List<CqnDelete> adaptActiveEntity(CqnDelete delete, EventContext context) {
		return adapt(delete, context, true);
	}

	/**
	 * Adapts the where condition and fields of the statement in order to allow queries
	 * using draft fields on the normal table/view that does not contain
	 * draft fields. To achieve this, path expressions to the sibling association
	 * are used as well as the replacement of some fields with default values,
	 * e.g. IsActiveEntity.
	 * @param stmt the statement
	 * @param context the {@link EventContext}
	 * @param <T> the type of the statement
	 * @return a statement that is semantically the same but can be executed on
	 *         the normal table/view
	 */
	public static <T extends CqnStatement> List<T> adapt(T stmt, EventContext context) {
		return adapt(stmt, context, false);
	}

	@SuppressWarnings("deprecation")
	private static <T extends CqnStatement> List<T> adapt(T stmt, EventContext context, boolean modifyIsActiveOnly) {
		AnalysisResult entityPath = CdsModelUtils.getEntityPath(stmt, context.getModel());
		// add draft associations
		if (stmt.isSelect() && stmt.asSelect().where().isPresent()
				&& context.getCdsRuntime().getEnvironment().getCdsProperties().getDrafts().getAssociationsToInactiveEntities().isEnabled()) {
			((Select<?>)stmt).where(addDraftAssociations(stmt.asSelect().where().get(), entityPath.targetEntity()));
		}
		List<List<Boolean>> followActiveAssociations = ReferenceModifier.getAssociationDirections(stmt, context);
		boolean addSecurityConstraints = stmt.isSelect();
		List<T> result = new ArrayList<>(followActiveAssociations.size());
		for (List<Boolean> followActive: followActiveAssociations) {
			ReferenceModifier modifier = new ReferenceModifier(entityPath.rootEntity(), modifyIsActiveOnly, followActive,
					addSecurityConstraints, context, !stmt.isInsert() && !stmt.isSelect());
			T toAdd = CQL.copy(stmt, modifier);
			AnalysisResult adaptedEntityPath = CdsModelUtils.getEntityPath(toAdd, context.getModel());
			if (stmt.isSelect()) {
				toAdd = CQL.copy(toAdd, new ColumnsModifier(adaptedEntityPath.targetEntity(), context));
			}
			if (!stmt.isInsert()) {
				toAdd = CQL.copy(toAdd, new DraftModifier(adaptedEntityPath.targetEntity(), !stmt.isSelect(), context));
			}
			result.add(toAdd);
		}
		return result;
	}

	public static Delete<?> toDelete(CqnSelect s) {
		Delete<StructuredType<?>> delete = Delete.from(s.ref());
		s.where().ifPresent(delete::where);
		return delete;
	}

	public static Select<?> toSelect(CqnDelete delete) {
		Select<?> select = Select.from(delete.ref());
		delete.where().ifPresent(select::where);
		return select;
	}

	/**
	 * Adds the draft associations for where conditions.
	 * @param predicate The original predicate
	 * @param entity the entity
	 * @return the
	 */
	private static CqnPredicate addDraftAssociations(CqnPredicate predicate, CdsEntity entity) {
		CqnPredicate result = predicate;
		if (predicate instanceof CqnConnectivePredicate) {
			CqnConnectivePredicate connective = (CqnConnectivePredicate) predicate;
			List<CqnPredicate> predicates = new ArrayList<>(connective.predicates());
			for (int i = 0; i < predicates.size(); ++i) {
				CqnPredicate replacement = addDraftAssociations(predicates.get(i), entity);
				predicates.remove(i);
				predicates.add(i, replacement);
			}
			result = CQL.connect(connective.operator(), predicates);
		} else if (predicate instanceof CqnComparisonPredicate) {
			CqnComparisonPredicate compPred = (CqnComparisonPredicate) predicate;
			List<CqnValue> leftValues;
			List<CqnValue> rightValues;
			if (compPred.left() instanceof ElementRef) {
				leftValues = resolveDraftAssociations((ElementRef<?>) compPred.left(), entity);
			} else {
				leftValues = Collections.singletonList(compPred.left());
			}
			if (compPred.right() instanceof ElementRef) {
				rightValues = resolveDraftAssociations((ElementRef<?>) compPred.right(), entity);
			} else {
				rightValues = Collections.singletonList(compPred.right());
			}
			List<CqnPredicate> newPredicates = new ArrayList<>(leftValues.size()*rightValues.size());
			Operator operator = compPred.operator();
			for (CqnValue leftValue: leftValues) {
				for (CqnValue rightValue: rightValues) {
					newPredicates.add(CQL.comparison(leftValue, operator, rightValue));
				}
			}
			result = CQL.or(newPredicates);
		} else if (predicate instanceof CqnNegation) {
			CqnNegation negation = (CqnNegation) predicate;
			result = CQL.not(addDraftAssociations(negation.predicate(), entity));
		} else if (predicate instanceof Xpr) {
			// TODO use methods once cds4j provides them
			Xpr plainPredicate = (Xpr) predicate;
			List<CqnToken> tokens = plainPredicate.xpr();
			CqnValue val = (CqnValue) tokens.remove(0);
			List<CqnValue> newValues;
			if (val instanceof ElementRef) {
				newValues = resolveDraftAssociations((ElementRef<?>) val, entity);
			} else {
				newValues = Collections.singletonList(val);
			}
			List<CqnPredicate> newPredicates = new ArrayList<>(newValues.size());
			for (CqnValue value : newValues) {
				List<CqnToken> p = new ArrayList<>(tokens);
				p.add(0, value);
				newPredicates.add(ExpressionBuilder.create(p).predicate());
			}
			result = CQL.or(newPredicates);
		}
		return result;
	}

	/**
	 * resolves all paths to draft entities
	 * @param ref the reference
	 * @param entity the entity
	 * @return the list containing all paths to draft entities
	 */
	private static List<CqnValue> resolveDraftAssociations(ElementRef<?> ref, CdsEntity entity) {
		List<Segment> segments = new ArrayList<>(ref.segments());
		List<List<Segment>> allPaths = new ArrayList<>();
		allPaths.add(segments);
		resolveDraftAssociations(segments, allPaths, 0, entity);
		List<CqnValue> result = new ArrayList<>();
		for (List<Segment> path: allPaths) {
			result.add(CQL.get(path));
		}
		return result;
	}

	private static void resolveDraftAssociations(List<Segment> segments, List<List<Segment>> allPaths, int j, CdsEntity entity) {
		CdsEntity targetEntity = entity;
		for (int i = j; i < segments.size(); ++i) {
			String id = segments.get(i).id();
			if (targetEntity.findAssociation(id).isPresent()) {
				String draftAssociation = id + DRAFT_SUFFIX;
				boolean draftAssociationFound = targetEntity.findAssociation(draftAssociation).isPresent();
				targetEntity = targetEntity.getTargetOf(id);
				if (DraftUtils.isDraftEnabled(targetEntity) && draftAssociationFound) {
					List<Segment> newSegments = new ArrayList<>(segments);
					Segment segment = newSegments.remove(i);
					newSegments.add(i, CQL.refSegment(draftAssociation, segment.filter().orElse(null)));
					allPaths.add(newSegments);
					resolveDraftAssociations(newSegments, allPaths, i, targetEntity);
				}
			}
		}
	}

	public static CqnPredicate getSecurityConstraints(EventContext context) {
		if (!context.getCdsRuntime().getEnvironment().getCdsProperties().getSecurity().getAuthorization().getDraftProtection().isEnabled()) {
			return null;
		}
		if (context.getUserInfo().isPrivileged()) { // in privileged mode drafts are always visible
			return null;
		}
		return Select.from("").where(c -> c.exists(outer -> Select.from(DraftAdministrativeData.CDS_NAME)
				.where(
						e -> e.get(DraftAdministrativeData.DRAFT_UUID).eq(outer.get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
						.and(
								CQL.or(
										e.get(DraftAdministrativeData.CREATED_BY_USER).isNull()
										, CQL.or(
												e.get(DraftAdministrativeData.CREATED_BY_USER).eq(context.getUserInfo().getId())
												, e.get(DraftAdministrativeData.CREATED_BY_USER).eq(context.getUserInfo().getName())
												)
										)
								)
						)
				)
				).where().get();
	}

	/**
	 * Replaces all references to {@code IsActiveEntity} with {@code value}
	 * @param delete the statement to adapt
	 * @param value the value for replacement
	 * @return the adapted statement
	 */
	public static CqnDelete replaceIsActiveEntity(CqnDelete delete, boolean value) {
		return CQL.copy(delete, new LeanModifier() {

			@Override
			public CqnStructuredTypeRef ref(CqnStructuredTypeRef ref) {
				RefBuilder<StructuredTypeRef> copy = CQL.copy(ref);
				copy.segments().forEach(s -> s.filter().ifPresent(f -> s.filter(CQL.copy(f, this))));
				return copy.build();
			}

			@Override
			public CqnValue ref(CqnElementRef ref) {
				if (Drafts.IS_ACTIVE_ENTITY.equals(ref.lastSegment())) {
					return CQL.val(value);
				}
				return ref;
			}
		});
	}

}
