/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ErrorStatuses;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.impl.utils.DeveloperModeUtils;
import com.sap.cds.services.messages.MessageTarget;
import com.sap.cds.services.utils.ErrorStatusException;

/**
 *
 * This class extends {@link ServiceException} and is supposed to be used in the
 * dispatch loop of the generic Service implementation. It adds EventContext
 * information to the exception chain.
 * <p>
 * Using the {@link ServiceException} should be considered in all other places.
 *
 */
public class ContextualizedServiceException extends ServiceException {

	private static final long serialVersionUID = 1L;

	private final EventContext errorContext;
	private final List<EventContext> viaList = new ArrayList<>();

	public ContextualizedServiceException(EventContext context, Throwable cause) {
		// a contextualized exception always has a null message
		super(null, cause);
		this.errorContext = context;
	}

	public void via(EventContext context) {
		viaList.add(context);
	}

	@Override
	public String getMessage() {
		StringBuilder builder = new StringBuilder();

		String nearestMessage = getNearest((t) -> t.getMessage());
		builder.append(nearestMessage == null ? "<no message>" : nearestMessage);
		if(errorContext != null) {
			Service service = errorContext.getService();
			String event = errorContext.getEvent();
			CdsEntity entity = errorContext.getTarget();
			builder.append(" (service '").append(service == null ? "<no service>" : service.getName()).append("', ");
			builder.append("event '").append(event == null ? "<no event>" : event).append("', ");
			builder.append("entity '").append(entity == null ? "<no entity>" : entity.getQualifiedName()).append("')");
		}

		if(Utils.getErrorsProperties().isExtended()) {
			DeveloperModeUtils.extendMessage(builder, this);
		}

		return builder.toString();
	}

	@Override
	public String getLocalizedMessage(Locale locale) {
		return getNearest((t) -> {
			if(t instanceof ServiceException) {
				return ((ServiceException) t).getLocalizedMessage(locale);
			} else if(t.getLocalizedMessage() != null) {
				boolean stackMessages = Utils.getErrorsProperties().getStackMessages().isEnabled();
				if(!stackMessages) {
					return new ErrorStatusException(ErrorStatuses.SERVER_ERROR).getLocalizedMessage(locale);
				} else {
					return t.getLocalizedMessage();
				}
			} else {
				return null;
			}
		});
	}

	@Override
	public String getPlainMessage() {
		return getNearest((t) -> {
			if(t instanceof ServiceException) {
				return ((ServiceException) t).getPlainMessage();
			} else {
				return t.getMessage();
			}
		});
	}

	@Override
	public ErrorStatus getErrorStatus() {
		ErrorStatus theErrorStatus = getNearest((t) -> {
			if(t instanceof ServiceException) {
				return ((ServiceException) t).getErrorStatus();
			}
			return null;
		});
		return theErrorStatus == null ? ErrorStatuses.SERVER_ERROR : theErrorStatus;
	}

	@Override
	public MessageTarget getMessageTarget() {
		return getNearest((t) -> {
			if(t instanceof ServiceException) {
				return ((ServiceException) t).getMessageTarget();
			}
			return null;
		});
	}

	public EventContext getErrorContext() {
		return errorContext;
	}

	public List<EventContext> getViaList() {
		return viaList; // NOSONAR
	}

	@Override
	protected List<EventContext> collectEventContexts() {
		ContextualizedServiceException previous = null;
		Throwable cause = getCause();
		while (previous == null && cause != null) {
			if (cause instanceof ContextualizedServiceException) {
				previous = (ContextualizedServiceException) cause;
			}
			cause = cause.getCause();
		}

		List<EventContext> result = new ArrayList<>();
		if (previous != null) {
			result.addAll(previous.collectEventContexts());
		}
		result.add(errorContext);
		result.addAll(viaList);
		return result;
	}

	private <T> T getNearest(Function<Throwable, T> provider) {
		T value = null;
		Throwable cause = getCause();
		while (value == null && cause != null) {
			if (!(cause instanceof ContextualizedServiceException)) {
				value = provider.apply(cause);
			}
			cause = cause.getCause();
		}
		return value;
	}
}
