package com.sap.cds.services.impl.draft;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.Row;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.impl.DeleteBuilder;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.cds.CapabilitiesHandler;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.model.CqnUtils;

/**
 * Event handlers for draft-specific actions.
 */
@ServiceName(value = "*", type = DraftService.class)
public class DraftActionsHandler implements EventHandler {

	@Before
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkCapabilityNewDraft(DraftNewEventContext context) {
		if(!CapabilitiesHandler.getCapabilities(context).isInsertable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INSERTABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkCapabilityEditDraft(DraftEditEventContext context) {
		if(!CapabilitiesHandler.getCapabilities(context).isUpdatable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_UPDATABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = {DraftService.EVENT_DRAFT_SAVE, DraftService.EVENT_DRAFT_EDIT, DraftService.EVENT_DRAFT_PREPARE})
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	protected void checkRootDraft(EventContext context) {
		if (!DraftUtils.isDraftEnabled(context.getTarget())) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_DRAFT_ENABLED, context.getTarget().getQualifiedName());
		}
		if (!DraftUtils.isDraftRoot(context.getTarget())) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_ROOT, context.getTarget().getQualifiedName());
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	protected Result defaultRead(CdsReadEventContext context) {
		return DraftReader.create(context).executeMerged(context.getCqn(), context.getCqnNamedValues());
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultPrepare(DraftPrepareEventContext context) {
		Result result = context.getService().run(context.getCqn());
		if (result.rowCount() == 0) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
					context.getTarget().getQualifiedName(), com.sap.cds.services.utils.model.CdsModelUtils
					.getTargetKeysAsString(context.getModel(), context.getCqn()));
		}
		// Just some basic check. The actual implementation is done by the application
		result.forEach(r -> {
			if (!Boolean.FALSE.equals(r.get(Drafts.IS_ACTIVE_ENTITY))) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INACTIVE);
			}
		});
		return result;
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultSave(DraftSaveEventContext context) {
		// Get entities to save
		CqnSelect select = Select.copy(context.getCqn())
				.columns(expandCompositions(context.getTarget()));

		// TODO should this read via the service READ event? Has implications on AuditLog!
		Result entriesToSave = DraftReader.create(context).executeMerged(select, context.getCqnNamedValues());

		// restore null values in the result, because they are not returned by cds4j
		CdsDataProcessor.create().addGenerator((p, e, t) -> t.isSimple(), (p, e, n) -> null).process(entriesToSave, context.getTarget());

		if (entriesToSave.rowCount() == 0) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
					context.getTarget().getQualifiedName(),
					com.sap.cds.services.utils.model.CdsModelUtils.getTargetKeysAsString(context.getModel(), select));
		}

		List<Row> newEntities = new ArrayList<>();
		List<Row> existingEntities = new ArrayList<>();
		entriesToSave.forEach(r -> {
			if (!Boolean.FALSE.equals(r.get(Drafts.IS_ACTIVE_ENTITY))) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INACTIVE);
			}
			if ((Boolean) r.get(Drafts.HAS_ACTIVE_ENTITY)) {
				existingEntities.add(r);
			} else {
				newEntities.add(r);
			}
		});

		// delete the draft entities
		// TODO consider doing that via the DraftService
		// however some applications rely on the current behaviour
		Result deleteResult;
		CqnDelete delete = CqnAdapter.create(context).adaptForInactiveExecution(CqnUtils.toDelete(select));
		if (delete == null) {
			deleteResult = DraftHandlerUtils.buildNoOpResult(context);
		} else {
			delete = CqnUtils.addWhere(delete, DraftModifier.getSecurityConstraints(context));
			List<Map<String, Object>> cqnValueSets = context.getCqnNamedValues() != null ? Arrays.asList(context.getCqnNamedValues()): Collections.emptyList();
			deleteResult = CdsServiceUtils.getDefaultPersistenceService(context).run(delete, cqnValueSets);
		}

		if (deleteResult.rowCount() != entriesToSave.rowCount()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INACTIVE);
		}

		List<Row> rows = new ArrayList<>();
		if (!newEntities.isEmpty()) {
			// insert new active entities
			CqnInsert insert = Insert.into(context.getTarget()).entries(newEntities);
			context.getService().run(insert).stream().forEach(rows::add);
		}
		if (!existingEntities.isEmpty()) {
			// update the active entities
			CqnUpdate update = Update.entity(context.getTarget()).entries(existingEntities);
			context.getService().run(update).stream().forEach(rows::add);
		}
		rows.forEach(row -> removeCompositionsAndAssociations(row, context.getTarget()));
		return ResultBuilder.insertedRows(rows).result();
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultEdit(DraftEditEventContext context) {
		// Get active entities
		CqnSelect select = Select.copy(context.getCqn())
				.columns(expandCompositions(context.getTarget()));

		// ensure that we read the data for edit without a locale!
		Result entriesToEdit = context.getCdsRuntime().requestContext().modifyParameters(param -> {
			param.setLocale(null);
		}).run( reqContext -> {
			// TODO should this read via the service READ event? Has implications on AuditLog!
			return DraftReader.create(context).executeMerged(select, context.getCqnNamedValues());
		});

		if (entriesToEdit.rowCount() == 0) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_INSTANCE_NOT_FOUND,
					context.getTarget().getQualifiedName(),
					com.sap.cds.services.utils.model.CdsModelUtils.getTargetKeysAsString(context.getModel(), select));
		}
		entriesToEdit.forEach(r -> {
			if (!Boolean.TRUE.equals(r.get(Drafts.IS_ACTIVE_ENTITY))) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_ACTIVE);
			}
			// clear on insert and on update fields, to ensure they are set after edit to a new value
			clearOnUpdateFields(r, context.getTarget());
		});

		// Cancel existing draft
		if (context.getPreserveChanges() != null && !context.getPreserveChanges()) {
			Delete<?> delete = CqnUtils.toDelete(context.getCqn());
			Instant timeoutThreshold = DraftModifier.getCancellationThreshold(context);
			Predicate timeoutCondition = CQL.exists(
				Select.from(DraftAdministrativeData.CDS_NAME).where(
					e -> e.get(DraftAdministrativeData.DRAFT_UUID).eq(CQL.to(CqnExistsSubquery.OUTER).get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
					.and(e.get(DraftAdministrativeData.LAST_CHANGE_DATE_TIME).le(timeoutThreshold))
				)
			);
			((DeleteBuilder<?>) delete).filter(timeoutCondition);
			// adapt for active execution to strip any IsActiveEntity = true condition from the statement
			CqnDelete deleteStatement = CqnAdapter.create(context).adaptForActiveExecution(delete);
			context.getCdsRuntime().requestContext().privilegedUser().run((requestContext) -> {
				context.getService().cancelDraft(deleteStatement);
			});
		}

		// Insert the entities into the drafts table
		CqnInsert insert = Insert.into(context.getTarget().getQualifiedName()).entries(entriesToEdit);
		Result result = ((DraftServiceImpl)context.getService()).createDraft(insert, true);
		return result;
	}

	private static void clearOnUpdateFields(Map<String, Object> m, CdsEntity entity) {
		CdsModelUtils.visitDeep(entity, m, (cdsEntity, data, parent, parentData) -> {
			cdsEntity.elements()
			.filter(e -> CdsAnnotations.ON_UPDATE.getOrDefault(e) != null)
			.map(e -> e.getName()).forEach(e -> data.remove(e));
		});
	}

	@SuppressWarnings("unchecked")
	private static <T> Function<T, CqnSelectListItem>[] expandCompositions(CdsEntity entity) {
		List<Function<StructuredType<?>, CqnSelectListItem>> columns = new ArrayList<>();
		columns.add(c -> c._all());
		entity.compositions().forEach(co -> columns.add(c -> c.to(co.getName()).expand(expandCompositions(entity.getTargetOf(co.getName())))));
		return columns.toArray(new Function[columns.size()]);
	}

	private static void removeCompositionsAndAssociations(Row r, CdsEntity entity) {
		Iterator<Map.Entry<String, Object>> iter = r.entrySet().iterator();
		while(iter.hasNext()) {
			Map.Entry<String, Object> entry = iter.next();
			if ((entry.getValue() instanceof List || entry.getValue() instanceof Map)
					&& entity.findAssociation(entry.getKey()).isPresent()) {
				iter.remove();
			}
		}
	}

}
