/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.application.ApplicationStoppedEventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.DraftGcEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.environment.CdsProperties.Drafts.GC;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = ApplicationLifecycleService.DEFAULT_NAME)
public class DraftGCHandler implements EventHandler {

	private static final Logger log = LoggerFactory.getLogger(DraftGCHandler.class);

	private final CdsRuntime runtime;
	private Timer timer;

	public DraftGCHandler(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@On
	protected void initializeGC(ApplicationPreparedEventContext context) {
		if (runtime.getEnvironment().getCdsProperties().getEnvironment().getCommand().isEnabled()) {
			return;
		}

		GC gcConfig = runtime.getEnvironment().getCdsProperties().getDrafts().getGc();
		if (gcConfig.isEnabled() && timer == null) {
			timer = new Timer("Draft GC Timer", true);
			long gcInterval = gcConfig.getInterval().toMillis();
			// distribute the first execution time randomly to reduce the database load when e.g. restarting the app
			long firstExecutionTime = ThreadLocalRandom.current().nextLong(gcInterval);
			timer.schedule(new TimerTask() {
				@Override
				public void run() {
					try {
						DraftUtils.gcDraftsOfAllServicesAndTenants(runtime);
					} catch (Throwable t) {
						log.error("Failed to gc drafts", t);
					}
				}
			}, firstExecutionTime, gcInterval);
		}
	}

	@On(service = "*", serviceType = DraftService.class)
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected void onGcDrafts(DraftGcEventContext context) {
		AtomicLong numCancelledDrafts = new AtomicLong(0);
		Instant threshold = Instant.now().minus(context.getCdsRuntime().getEnvironment().getCdsProperties().getDrafts().getDeletionTimeout()).truncatedTo(ChronoUnit.MILLIS);
		DraftService draftService = context.getService();
		draftService.getDefinition().entities().forEach(e -> {
			if (DraftUtils.isDraftRoot(e) && !e.getQualifiedName().endsWith(DraftModifier.DRAFT_SUFFIX)) {
				CqnDelete deleteOldDrafts = Delete.from(e).where(c -> c.get(Drafts.IS_ACTIVE_ENTITY).eq(false).and(c.exists(
						outer -> Select.from(DraftAdministrativeData.CDS_NAME)
						.where(a -> a.get(DraftAdministrativeData.DRAFT_UUID).eq(outer.get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
								.and(a.get(DraftAdministrativeData.LAST_CHANGE_DATE_TIME).le(threshold))))));
				Result result = context.getCdsRuntime().requestContext().privilegedUser().run((requestContext) -> {
					return draftService.cancelDraft(deleteOldDrafts);
				});
				if (result.rowCount() > 0) {
					log.info("Draft GC deleted {} drafts of entity '{}'", result.rowCount(), e.getQualifiedName());
					numCancelledDrafts.addAndGet(result.rowCount());
				}
			}
		});
		context.setResult(ResultBuilder.deletedRows(numCancelledDrafts.get()).result());
	}

	@On
	protected void stopGC(ApplicationStoppedEventContext context) {
		if(timer != null) {
			timer.cancel();
			timer = null;
		}
	}

}
