package com.sap.cds.services.impl.draft;

import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.stream.Collectors;

import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.Row;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.impl.SelectBuilder;
import com.sap.cds.ql.impl.SelectListValueBuilder;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CqnStatementUtils;

public class DraftReader {

	private static final String ADDED_COLUMN_PREFIX = "ADDED_FOR_DRAFT_SORTING_";

	private final EventContext context;

	public static DraftReader create(EventContext context) {
		return new DraftReader(context);
	}

	private DraftReader(EventContext context) {
		this.context = context;
	}

	/**
	 * Executes the {@link CqnSelect} against the active and draft persistence and merges the results.
	 * @param select the {@link CqnSelect}
	 * @param cqnNamedValues the statement parameters
	 * @return the merged {@link Result}
	 */
	public Result executeMerged(CqnSelect select, Map<String, Object> cqnNamedValues) {
		if (select.orderBy().isEmpty() && !select.hasLimit()) {
			return executeMergedSimple(select, cqnNamedValues);
		} else {
			return executeMergedOrderedAndPaged(select, cqnNamedValues);
		}
	}

	private Result executeMergedSimple(CqnSelect select, Map<String, Object> cqnNamedValues) {
		Result resultActive = ((DraftServiceImpl) context.getService()).readActive(select, cqnNamedValues);
		Result resultInactive = ((DraftServiceImpl) context.getService()).readDraft(select, cqnNamedValues);
		List<Row> merged = new ArrayList<>();
		resultActive.stream().forEach(merged::add);
		resultInactive.stream().forEach(merged::add);
		return ResultBuilder.selectedRows(merged).inlineCount(calcInlineCount(resultActive, resultInactive)).rowType(resultActive.rowType()).result();
	}

	private Result executeMergedOrderedAndPaged(CqnSelect select, Map<String, Object> cqnNamedValues) {
		// determine row type, before adding items for sorting
		CdsStructuredType rowType = CqnStatementUtils.rowType(context.getModel(), select);
		LinkedHashMap<CqnSortSpecification, String> sortSpecificationToPath = ensureOrderByInColumns(select);
		Select<?> active = SelectBuilder.copyShallow(select);
		Select<?> inactive = SelectBuilder.copyShallow(select);

		// we assume a low number of draft entities
		inactive.limit(-1, 0);

		Result resultInactive = ((DraftServiceImpl) context.getService()).readDraft(inactive, cqnNamedValues);
		long skipDiff = 0;
		long originalTop = select.top() < 0 ? Long.MAX_VALUE : select.top();
		if (select.hasLimit()) {
			// we have to select more active entities to sort in the draft entities correctly
			long draftsRead = resultInactive.rowCount();
			long originalSkip = select.skip();
			long newSkip = Math.max(0, originalSkip - draftsRead);
			long newTop = Math.max(originalTop + draftsRead, originalTop);
			skipDiff = originalSkip - newSkip;
			active.limit(newTop, newSkip);
		}
		Result resultActive = ((DraftServiceImpl) context.getService()).readActive(active, cqnNamedValues);
		List<Row> sortedRows = sortResults(resultActive, resultInactive, sortSpecificationToPath);
		if (select.hasLimit()) {
			long newEnd = Math.max(originalTop + skipDiff, originalTop);
			long end = Math.min(newEnd, sortedRows.size());
			sortedRows = sortedRows.subList((int) skipDiff, (int) end);
		}

		if (sortSpecificationToPath.values().stream().anyMatch(value -> value.startsWith(ADDED_COLUMN_PREFIX))) {
			sortedRows.forEach(row -> row.keySet().removeIf(key -> key.startsWith(ADDED_COLUMN_PREFIX)));
		}
		return ResultBuilder.selectedRows(sortedRows).inlineCount(calcInlineCount(resultActive, resultInactive)).rowType(rowType).result();
	}

	private LinkedHashMap<CqnSortSpecification, String> ensureOrderByInColumns(CqnSelect select) {
		// This Map will index each column of the Select to the alias except columns that are not of CqnElementRef
		// Certain combination of expands may lead to same items being in the select -> the collector will merge them
		Map<String, String> selectColumnsToSearchableNames = CqnStatementUtils.resolveStar(select, context.getTarget())
				.items().stream()
				.flatMap(CqnSelectListItem::ofRef)
				.collect(Collectors.toMap(CqnElementRef::path, CqnSelectListValue::displayName, (s, s2) -> s2));

		LinkedHashMap<CqnSortSpecification, String> pathLookup = new LinkedHashMap<>(select.orderBy().size());
		List<CqnSelectListItem> newItems = new ArrayList<>();

		int i = 0;
		for (CqnSortSpecification sortSpec : select.orderBy()) {
			CqnValue val = sortSpec.value();
			String displayName = val.isRef() ? selectColumnsToSearchableNames.get(val.asRef().path()) : null;
			// value used in orderby is not selected
			if (displayName == null) {
				displayName = ADDED_COLUMN_PREFIX + (++i);
				newItems.add(SelectListValueBuilder.select(val).as(displayName).build());
			}
			pathLookup.put(sortSpec, displayName);
		}

		if (!newItems.isEmpty()) {
			newItems.addAll(0, select.items().isEmpty() ? Collections.singletonList(CQL.star()) : select.items());
			((Select<?>) select).columns(newItems);
		}
		return pathLookup;
	}

	@SuppressWarnings("unchecked")
	private List<Row> sortResults(Result active, Result inactive, LinkedHashMap<CqnSortSpecification, String> sortSpecificationToPath) {
		// Assume that active entities and drafts are sorted separately, and we do not need to rearrange them again
		if (active.rowCount() == 0) {
			return inactive.list();
		} else if (inactive.rowCount() == 0) {
			return active.list();
		}
		List<Row> result = new ArrayList<>((int) (active.rowCount() + inactive.rowCount()));
		Iterator<Row> activeIter = active.iterator();
		Iterator<Row> draftIter = inactive.iterator();
		Row rActive = activeIter.hasNext() ? activeIter.next() : null;
		Row rDraft = draftIter.hasNext() ? draftIter.next() : null;

		Locale locale = context.getParameterInfo().getLocale();
		Collator collator = locale != null ? Collator.getInstance(locale) : null;
		Comparator<Row> cmp = (o1, o2) -> {
			if (o1 == null) {
				return 1;
			}
			if (o2 == null) {
				return -1;
			}
			for (Entry<CqnSortSpecification, String> entry : sortSpecificationToPath.entrySet()) {
				String item = entry.getValue();
				int orderFactor = entry.getKey().order() == CqnSortSpecification.Order.DESC ? -1 : 1;
				Object value1 = o1.getPath(item);
				Object value2 = o2.getPath(item);
				if (value1 == null && value2 == null) {
					continue;
				}

				// put null values always at the beginning of the list (if ascending order) to
				// maintain DB sort order
				if (value1 == null) {
					return -orderFactor;
				}
				if (value2 == null) {
					return orderFactor;
				}
				if (!(value1 instanceof Comparable) || !(value2 instanceof Comparable)) {
					throw new ErrorStatusException(CdsErrorStatuses.INVALID_SORT_ELEMENT, item,
							context.getTarget().getQualifiedName());
				}

				int comp;
				if ((value1 instanceof String) && (value2 instanceof String) && collator != null) {
					comp = collator.compare(value1, value2);
				} else {
					comp = ((Comparable<Object>) value1).compareTo(value2);
				}
				if (comp != 0) {
					return orderFactor * comp;
				}
			}
			// sort active and inactive entities always in the same order
			if (Boolean.TRUE.equals(o1.get(Drafts.IS_ACTIVE_ENTITY))) {
				return Boolean.TRUE.equals(o2.get(Drafts.IS_ACTIVE_ENTITY)) ? 0 : 1;
			} else {
				return Boolean.TRUE.equals(o2.get(Drafts.IS_ACTIVE_ENTITY)) ? -1 : 0;
			}
		};

		while (rActive != null || rDraft != null) {
			int comparison = Objects.compare(rDraft, rActive, cmp);

			if (comparison > 0) {
				result.add(rActive);
				rActive = activeIter.hasNext() ? activeIter.next() : null;
			} else {
				result.add(rDraft);
				rDraft = draftIter.hasNext() ? draftIter.next() : null;
			}
		}
		return result;
	}

	private static long calcInlineCount(Result r1, Result r2) {
		if (r1.inlineCount() >= 0 || r2.inlineCount() >= 0) {
			return Math.max(0, r1.inlineCount()) + Math.max(0, r2.inlineCount());
		}
		return -1;
	}

}
