/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.handlerregistry.resolver;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;
import com.sap.cds.CdsData;
import com.sap.cds.Result;
import com.sap.cds.ql.CdsName;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.DraftCancelEventContext;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPatchEventContext;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.impl.draft.ActiveReadEventContext;
import com.sap.cds.services.impl.draft.DraftReadEventContext;
import com.sap.cds.services.impl.draft.DraftServiceImpl;
import com.sap.cds.services.impl.handlerregistry.AnnotationDescriptor;
import com.sap.cds.services.impl.handlerregistry.HandlerDescriptor;
import com.sap.cds.services.impl.utils.ReflectionUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.ResultUtils;

/**
 * Handles all types of CDS Model POJO based arguments in handler methods. It supports {@link List}, {@link Stream} and direct usage of POJOs
 */
public class PojoArgumentResolver implements ArgumentResolver {

	private final static Logger logger = LoggerFactory.getLogger(PojoArgumentResolver.class);
	private final static Set<String> allowedEvents = Sets.newHashSet(CqnService.EVENT_READ
			, CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_DELETE, CqnService.EVENT_UPSERT
			, DraftService.EVENT_DRAFT_EDIT, DraftService.EVENT_DRAFT_PREPARE, DraftService.EVENT_DRAFT_SAVE
			, DraftService.EVENT_DRAFT_NEW, DraftService.EVENT_DRAFT_PATCH, DraftService.EVENT_DRAFT_CANCEL
			, DraftService.EVENT_DRAFT_CREATE, DraftServiceImpl.EVENT_DRAFT_READ, DraftServiceImpl.EVENT_ACTIVE_READ);
	private final static HashMap<Type, PojoArgumentResolver> cache = new HashMap<>();

	private static enum ArgumentType {
		SINGLE,
		STREAM,
		LIST,
		RESULT
	}

	/**
	 * Returns a {@link PojoArgumentResolver}, if the class and generic type are applicable for this resolver
	 * @param clazz the class of the argument
	 * @param genericType the generic argument type
	 * @return A {@link PojoArgumentResolver}, or null, if the resolver is not applicable.
	 */
	public static PojoArgumentResolver createIfApplicable(Class<?> clazz, Type genericType) {
		PojoArgumentResolver cached;
		if((cached = cache.get(genericType)) != null) {
			return cached;
		}

		try {
			if(isPojoClass(clazz)) {
				return new PojoArgumentResolver(clazz, genericType, ArgumentType.SINGLE);
			} else if (Result.class.isAssignableFrom(clazz)) {
				return new PojoArgumentResolver(CdsData.class, genericType, ArgumentType.RESULT);
			} else {
				boolean stream = Stream.class.isAssignableFrom(clazz);
				boolean list = List.class.isAssignableFrom(clazz);
				if (stream || list) {
					ParameterizedType collectionType = ReflectionUtils.getParameterizedType(genericType);
					if(collectionType.getActualTypeArguments().length == 1) {
						Class<?> pojoClass = ReflectionUtils.getClassForType(collectionType.getActualTypeArguments()[0]);
						if(isPojoClass(pojoClass)) {
							return new PojoArgumentResolver(pojoClass, genericType, stream ? ArgumentType.STREAM : ArgumentType.LIST);
						}
					}
				}
			}
		} catch (IllegalArgumentException | ClassNotFoundException e) {
			logger.debug("Could not resolve generic type '{}'", genericType.getTypeName(), e);
		}

		return null;
	}

	/**
	 * True, if the type is of a CSD Model POJO class.
	 * @param type the type
	 * @return true, if the type is of a CDS Model POJO class.
	 * @throws ClassNotFoundException
	 */
	private static boolean isPojoClass(Class<?> pojoClass) throws ClassNotFoundException {
		return CdsData.class.isAssignableFrom(pojoClass);
	}

	private final Class<?> pojoClass;
	private final Type genericType;
	private final ArgumentType argumentType;

	private PojoArgumentResolver(Class<?> pojoClass, Type genericType, ArgumentType argumentType) {
		this.pojoClass = pojoClass;
		this.genericType = genericType;
		this.argumentType = argumentType;
		cache.put(genericType, this);
	}

	@Override
	public Object resolve(EventContext context) {
		if(context.isCompleted()) {
			// go to result in AFTER phase
			switch (context.getEvent()) {
			case CqnService.EVENT_READ:
				return convert(context.as(CdsReadEventContext.class).getResult());
			case CqnService.EVENT_CREATE:
				return convert(context.as(CdsCreateEventContext.class).getResult());
			case CqnService.EVENT_UPDATE:
				return convert(context.as(CdsUpdateEventContext.class).getResult()); // might be only partial
			case CqnService.EVENT_DELETE:
				return convert(context.as(CdsDeleteEventContext.class).getResult()); // no data available by default, but custom ON handler might add it
			case CqnService.EVENT_UPSERT:
				return convert(context.as(CdsUpsertEventContext.class).getResult());
			case DraftService.EVENT_DRAFT_NEW:
				return convert(context.as(DraftNewEventContext.class).getResult());
			case DraftService.EVENT_DRAFT_EDIT:
				return convert(context.as(DraftEditEventContext.class).getResult());
			case DraftService.EVENT_DRAFT_PATCH:
				return convert(context.as(DraftPatchEventContext.class).getResult());
			case DraftService.EVENT_DRAFT_PREPARE:
				return convert(context.as(DraftPrepareEventContext.class).getResult());
			case DraftService.EVENT_DRAFT_SAVE:
				return convert(context.as(DraftSaveEventContext.class).getResult());
			case DraftService.EVENT_DRAFT_CANCEL:
				return convert(context.as(DraftCancelEventContext.class).getResult());
			case DraftService.EVENT_DRAFT_CREATE:
				return convert(context.as(DraftCreateEventContext.class).getResult());
			case DraftServiceImpl.EVENT_DRAFT_READ:
				return convert(context.as(DraftReadEventContext.class).getResult());
			case DraftServiceImpl.EVENT_ACTIVE_READ:
				return convert(context.as(ActiveReadEventContext.class).getResult());
			default:
				return null;
			}
		} else if (argumentType != ArgumentType.RESULT) {
			// go to CQN in BEFORE / ON phase
			switch (context.getEvent()) {
			case CqnService.EVENT_CREATE:
				return convert(ResultUtils.convert(context.as(CdsCreateEventContext.class).getCqn().entries()));
			case CqnService.EVENT_UPDATE:
				return convert(ResultUtils.convert(context.as(CdsUpdateEventContext.class).getCqn().entries())); // might be only partial
			case CqnService.EVENT_UPSERT:
				return convert(ResultUtils.convert(context.as(CdsUpsertEventContext.class).getCqn().entries()));
			case DraftService.EVENT_DRAFT_NEW:
				return convert(ResultUtils.convert(context.as(DraftNewEventContext.class).getCqn().entries()));
			case DraftService.EVENT_DRAFT_PATCH:
				return convert(ResultUtils.convert(context.as(DraftPatchEventContext.class).getCqn().entries())); // might be only partial
			case DraftService.EVENT_DRAFT_CREATE:
				return convert(ResultUtils.convert(context.as(DraftCreateEventContext.class).getCqn().entries()));
			default:
				return null;
			}
		}
		return null;
	}

	private Object convert(Result result) {
		switch(argumentType) {
			case RESULT:
				return result;
			case STREAM:
				return result.streamOf(pojoClass);
			case LIST:
				return result.listOf(pojoClass);
			case SINGLE:
				return result.single(pojoClass);
			default:
				throw new IllegalStateException();
		}
	}

	@Override
	public void verifyOrThrow(HandlerDescriptor descriptor) {
		Set<String> events = new HashSet<>();
		for(AnnotationDescriptor ad : descriptor.getAnnotations()) {
			events.addAll(Arrays.asList(ad.getEvents()));
		}

		for(String event : events) {
			verifyEventOrThrow(event, descriptor);
		}
	}

	private void verifyEventOrThrow(String event, HandlerDescriptor descriptor) {
		if(!allowedEvents.contains(event)) {
			throw new ErrorStatusException(CdsErrorStatuses.POJO_ARGUMENT_MISMATCH, event, genericType.getTypeName(), descriptor.getMethodName());
		}
	}

	@Override
	public String[] indicateEntities() {
		CdsName cdsName = pojoClass.getAnnotation(CdsName.class);
		if(cdsName != null) {
			return new String[] { cdsName.value() };
		}
		return new String[0];
	}

}
