/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.sap.cds.CdsData;
import com.sap.cds.Struct;
import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.CdsModelReader;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.auditlog.Action;
import com.sap.cds.services.auditlog.AuditLogService;
import com.sap.cds.services.auditlog.ConfigChangeLog;
import com.sap.cds.services.auditlog.ConfigChangeLogContext;
import com.sap.cds.services.auditlog.DataAccessLog;
import com.sap.cds.services.auditlog.DataAccessLogContext;
import com.sap.cds.services.auditlog.DataModificationLog;
import com.sap.cds.services.auditlog.DataModificationLogContext;
import com.sap.cds.services.auditlog.SecurityLog;
import com.sap.cds.services.auditlog.SecurityLogContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;

/**
 * An {@link EventHandler} to handle auditlog outbox related events.
 */
@ServiceName(value = "*", type = OutboxService.class  )
public class AuditLogDefaultOutboxOnHandler implements EventHandler {

	public static final String OUTBOX_AUDITLOG_TARGET = "auditlog/" + AuditLogService.DEFAULT_NAME;

	private static final JsonMapper mapper = new JsonMapper();

	private static final String USER = "user";
	private static final String TENANT = "tenant";
	private static final String LOGON_NAME = "logonName";
	private static final String EVENT = "event";

	// common properties from auditlog-related contexts
	private static final String DATA = "data";
	private static final String CREATED_AT = "createdAt";
	private static final String ACTION = "action";
	private static final String MODIFICATIONS = "modifications";

	private CdsModel auditlogModel;

	public AuditLogDefaultOutboxOnHandler() {
		// interim solution
		String csnPath = "com/sap/cds/auditlog.csn";
		try (InputStream is = AuditLogDefaultOutboxOnHandler.class.getClassLoader().getResourceAsStream(csnPath)) {
			if (is == null) {
				throw new ErrorStatusException(CdsErrorStatuses.NO_AUDITLOG_MODEL, csnPath);
			}
			auditlogModel = CdsModelReader.read(is);
		} catch (IOException e) {
			throw new ErrorStatusException(CdsErrorStatuses.AUDITLOG_NOT_READABLE, csnPath, e);
		}
	}

	@On(event = OUTBOX_AUDITLOG_TARGET)
	private void publishedByOutbox(OutboxMessageEventContext context) {
		Service auditlogService = context.getServiceCatalog().getService(AuditLogService.DEFAULT_NAME);
		if (auditlogService != null) {
			EventContextAccessor contextAccessor = new EventContextAccessor(context, auditlogModel);

			// emit in the user context
			context.getCdsRuntime().requestContext().modifyUser(u -> {
				u.setName(contextAccessor.getUser());

				// always use tenant provided by auditlog message creation moment
				u.setTenant(contextAccessor.getTenant());
				u.setAdditionalAttribute(LOGON_NAME, contextAccessor.getLogonName());
			}).run(req -> {
				auditlogService.emit(contextAccessor.getContext());
			});
		}
	}

	/**
	 * Serialization of the event context with CDS data element to the outbox entry.
	 *
	 * @param context event context
	 *
	 * @return outbox entry message
	 */
	static String createOutboxMessage(EventContext context, CdsRuntime runtime) {
		RequestContext requestContext = RequestContext.getCurrent(runtime);
		// handle the serialization
		try {
			CdsData data = (CdsData) context.get(DATA);

			Map<String, Object> map = new HashMap<>();
			map.put(USER, requestContext.getUserInfo().getName());
			map.put(TENANT, requestContext.getUserInfo().getTenant());
			String logonName = (String) requestContext.getUserInfo().getAdditionalAttribute(LOGON_NAME);
			if (!StringUtils.isEmpty(logonName)) {
				map.put(LOGON_NAME, logonName);
			}
			map.put(EVENT, context.getEvent());
			map.put(DATA, data.toJson());

			return mapper.writeValueAsString(map);
		} catch (Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.CONTEXT_SERIALIZATION_FAILED, context.getEvent(), e);
		}
	}

	private static class EventContextAccessor {

		private String user;
		private String tenant;
		private String logonName;

		private EventContext context;

		private EventContextAccessor(OutboxMessageEventContext outboxContext, CdsModel auditlogModel) {
			try {
				// deserialize the data
				Map<String, Object> contextData =  mapper.readValue(outboxContext.getMessage(), new TypeReference<Map<String, Object>>() {});

				this.user = (String) contextData.get(USER);
				this.tenant = (String) contextData.get(TENANT);
				this.logonName = (String) contextData.get(LOGON_NAME);

				String event = (String) contextData.get(EVENT);
				this.context =  EventContext.create(event, null);

				CdsStructuredType type = auditlogModel.getEvent("com.sap.cds.services.auditlog." + event);
				Map<String, Object> data = StructDataParser.create(type).parseObject((String) contextData.get(DATA));
				adaptActionType(data, event);

				setData(context, event, data);
				context.put(CREATED_AT, outboxContext.getTimestamp());
				context.put(OutboxService.IS_OUTBOXED, true);

			} catch (Exception e) {
				throw new ErrorStatusException(CdsErrorStatuses.CONTEXT_DESERIALIZATION_FAILED, outboxContext.getEvent(), e);
			}
		}

		private void setData(EventContext context, String event, Map<String, Object> data) {
			// set data using the typed interface according to the concrete event type
			if (event.equals(DataModificationLogContext.CDS_NAME)) {
				DataModificationLog log = Struct.access(data).as(DataModificationLog.class);
				context.as(DataModificationLogContext.class).setData(log);
			} else if (event.equals(ConfigChangeLogContext.CDS_NAME)) {
				ConfigChangeLog log = Struct.access(data).as(ConfigChangeLog.class);
				context.as(ConfigChangeLogContext.class).setData(log);
			} else if (event.equals(SecurityLogContext.CDS_NAME)) {
				SecurityLog log = Struct.access(data).as(SecurityLog.class);
				context.as(SecurityLogContext.class).setData(log);
			} else if (event.equals(DataAccessLogContext.CDS_NAME)) {
				DataAccessLog log = Struct.access(data).as(DataAccessLog.class);
				context.as(DataAccessLogContext.class).setData(log);
			}
		}

		@SuppressWarnings("unchecked")
		private void adaptActionType(Map<String, Object> data, String event) {
			// adapt the CDS types
			if (event.equals(DataModificationLogContext.CDS_NAME)) {
				Collection<Map<String, Object>> modifications = (Collection<Map<String, Object>>) ((Map<String, ?>) data).get(MODIFICATIONS);
				if (modifications != null) {
					modifications.forEach(modification -> {
						Object action = modification.get(ACTION);
						if (action != null) {
							modification.put(ACTION, Action.valueOf(action.toString()));
						}
					});
				}
			} else if (event.equals(ConfigChangeLogContext.CDS_NAME)) {
				Object action = data.get(ACTION);
				if (action != null) {
					data.put(ACTION, Action.valueOf(action.toString()));
				}
			}
		}

		public String getUser() {
			return user;
		}

		public String getTenant() {
			return tenant;
		}

		public String getLogonName() {
			return logonName;
		}

		public EventContext getContext() {
			return context;
		}
	}
}
