/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.List;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.ServiceDelegator;
import com.sap.cds.services.auditlog.Access;
import com.sap.cds.services.auditlog.Action;
import com.sap.cds.services.auditlog.AuditLogService;
import com.sap.cds.services.auditlog.ConfigChange;
import com.sap.cds.services.auditlog.ConfigChangeLog;
import com.sap.cds.services.auditlog.ConfigChangeLogContext;
import com.sap.cds.services.auditlog.DataAccessLog;
import com.sap.cds.services.auditlog.DataAccessLogContext;
import com.sap.cds.services.auditlog.DataModification;
import com.sap.cds.services.auditlog.DataModificationLog;
import com.sap.cds.services.auditlog.DataModificationLogContext;
import com.sap.cds.services.auditlog.SecurityLog;
import com.sap.cds.services.auditlog.SecurityLogContext;
import com.sap.cds.services.environment.CdsProperties.AuditLog;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.outbox.OutboxUtils;

/**
 * Implementation of {@link AuditLogService}.
 */
public class AuditLogServiceImpl extends ServiceDelegator implements AuditLogService {

	private final CdsRuntime runtime;

	private final AuditLog auditlogProperties;

	AuditLogServiceImpl(String name, CdsRuntime runtime) {
		super(name);
		this.runtime = runtime;
		this.auditlogProperties = runtime.getEnvironment().getCdsProperties().getAuditLog();
	}

	@Override
	public void logDataAccess(List<Access> accesses) {
		DataAccessLog log = DataAccessLog.create();
		log.setAccesses(accesses);

		DataAccessLogContext context = DataAccessLogContext.create();
		context.setData(log);

		emit(context);
	}

	@Override
	public void logDataModification(List<DataModification> modifications) {
		DataModificationLog log = DataModificationLog.create();
		log.setModifications(modifications);

		DataModificationLogContext context = DataModificationLogContext.create();
		context.setData(log);

		emit(context);
	}

	@Override
	public void logConfigChange(Action action, List<ConfigChange> configurations) {
		ConfigChangeLog changeLog = ConfigChangeLog.create();
		changeLog.setAction(action);
		changeLog.setConfigurations(configurations);

		ConfigChangeLogContext context = ConfigChangeLogContext.create();
		context.setData(changeLog);

		emit(context);
	}

	@Override
	public void logSecurityEvent(String action, String data) {
		SecurityLog log = SecurityLog.create();
		log.setAction(action);
		log.setData(data);

		SecurityLogContext context = SecurityLogContext.create();
		context.setData(log);

		emit(context);
	}

	@Override
	public void emit(EventContext context) {
		OutboxService outbox;
		// check whether the outbox may and should be used
		boolean persistentOutboxAllowed = auditlogProperties.getOutbox().getPersistent().isEnabled();
		if (shouldBeOutboxed(context) && (outbox = OutboxUtils.chooseOutboxService(runtime, persistentOutboxAllowed)) != null) {
			String message = AuditLogDefaultOutboxOnHandler.createOutboxMessage(context, runtime);
			outbox.enroll(AuditLogDefaultOutboxOnHandler.OUTBOX_AUDITLOG_TARGET, message);
		} else {
			super.emit(context);
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.AUTO_COMPLETE)
	private void autoComplete(EventContext context) {
		context.setCompleted();
	}

	private boolean shouldBeOutboxed(EventContext context) {
		return auditlogProperties.getOutbox().isEnabled()
				&& context.get(OutboxService.IS_OUTBOXED) == null
				&& !context.getEvent().equals(SecurityLogContext.CDS_NAME); // security events may not be outboxed at all
	}
}
