/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.messaging.file;

import static com.sap.cds.services.messaging.utils.MessagingUtils.toStructuredMessage;
import static com.sap.cds.services.messaging.utils.MessagingUtils.toStringMessage;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.impl.util.Pair;
import com.sap.cds.services.messaging.TopicMessageEventContext;
import com.sap.cds.services.messaging.service.MessagingBrokerQueueListener;
import com.sap.cds.services.messaging.service.MessagingBrokerQueueListener.MessageAccess;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;


/**
 * Implementation of the internal file based messaging service for local environment tests.
 */
public class FileBasedMessagingBroker {

	private static final Logger logger = LoggerFactory.getLogger(FileBasedMessagingBroker.class);

	private final String name;
	private final File file;
	private final Set<String> subscriptions = ConcurrentHashMap.newKeySet();
	private final boolean forceListening;
	private MessagingBrokerQueueListener listener;
	private WatchService fileWatcher;

	public FileBasedMessagingBroker(String name, File exchangeFile, boolean forceListening) throws IOException {
		this.name = name;
		this.file = new File(exchangeFile.getCanonicalPath());
		this.forceListening = forceListening;
	}

	/**
	 * Emits a new message to the file.
	 *
	 * @param topic the topic the message should be send to
	 * @param messageEventContext the TopicMessageEventContext containing the message that shall be send
	 */
	public void emitMessage(String topic, TopicMessageEventContext messageEventContext) {
		try (RandomAccessFile raFile = new RandomAccessFile(file, "rw");
				FileChannel channel = raFile.getChannel();
				FileLock lock = obtainFileLock(60000, channel)) {
			MessageLine line = new MessageLine(topic, toStringMessage(messageEventContext));
			channel.write(ByteBuffer.wrap(line.toString().getBytes(StandardCharsets.UTF_8)), channel.size());
		} catch (Exception e) { // NOSONAR
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_EMITTING_FAILED, topic, e);
		}
	}

	public void registerListener(MessagingBrokerQueueListener listener) throws IOException {
		if(this.listener != null) {
			throw new IllegalStateException("Only one listener is expected to be registered");
		}
		this.listener = listener;
		this.fileWatcher = FileSystems.getDefault().newWatchService();
		Path dir = Paths.get(file.getParentFile().getAbsolutePath());
		dir.register(fileWatcher, StandardWatchEventKinds.ENTRY_MODIFY);
		new Thread(() ->  {
			WatchKey watchKey = null;
			while (true) {
				try {
					watchKey = fileWatcher.poll(10, TimeUnit.MINUTES);
					if(watchKey != null) {
						if(!watchKey.pollEvents().isEmpty()) {
							receivedRawEvent();
						}
						watchKey.reset();
					}
				} catch (InterruptedException e) { // NOSONAR
					// This exception can be thrown by the fileWatcher.poll(). If it is thrown, we just start over. Hence it is not
					// an issue to just log the fact that the exception has been thrown.
					logger.debug("File watching for '{}' timed out, restarting watcher.", name);
				} catch (ClosedWatchServiceException e) {
					logger.debug("Stopped file watching for '{}'", name);
					break;
				}
			}
		}, "FileBasedBroker " + name).start();
	}

	public void subscribeTopic(String topic) {
		subscriptions.add(topic);
	}

	public void stop() {
		if (fileWatcher != null) {
			try {
				fileWatcher.close();
			} catch (IOException e) {
				logger.debug("Failed to stop file watching for '{}'", name);
			}
		}
	}

	/**
	 * Get notified when the file contents changed.
	 */
	private void receivedRawEvent() {
		List<MessageLine> toPublishLines = new ArrayList<>();
		try (RandomAccessFile raFile = new RandomAccessFile(file, "rw");
				FileChannel channel = raFile.getChannel();
				FileLock lock = obtainFileLock(60000, channel)) {
			MessageLine line;
			List<MessageLine> leftLines = new ArrayList<>();
			while ((line = readNextLine(raFile)) != null) {
				if (line.isEmpty()) {
					continue;
				}
				if (forceListening || subscriptions.contains(line.getBrokerTopic())) {
					toPublishLines.add(line);
				} else {
					leftLines.add(line);
				}
			}

			if (!toPublishLines.isEmpty()) {
				// cleanup the file
				raFile.setLength(0);
				for (MessageLine l : leftLines) {
					raFile.write(l.toString().getBytes(StandardCharsets.UTF_8));
				}
			}
		} catch (Exception e) { // NOSONAR
			logger.error("Could not read messages from '{}'", file, e);
		} finally {
			// notify listener if any available
			if(listener != null) {
				toPublishLines.forEach(message -> {
					try {
						listener.receivedMessage(message);
					} catch (Throwable e) { // NOSONAR
						logger.error("The received message with topic '{}' could not be handled", message.getBrokerTopic(), e);
					}
				});
			}
		}
	}

	private MessageLine readNextLine(RandomAccessFile file) throws IOException {
		String line = file.readLine();
		if (line != null) {
			return new MessageLine(new String(line.getBytes(StandardCharsets.ISO_8859_1), StandardCharsets.UTF_8));
		}
		return null;
	}

	private FileLock obtainFileLock(long timeout, FileChannel channel) throws IOException, InterruptedException, TimeoutException {
		Long quitTime = System.currentTimeMillis() + timeout;
		do {
			try {
				FileLock lock = channel.tryLock();
				if (lock != null) {
					return lock;
				} else {
					Thread.sleep(1000);
				}
			} catch (OverlappingFileLockException e) { // NOSONAR
				Thread.sleep(1000);
			}
		} while (System.currentTimeMillis() < quitTime);

		throw new TimeoutException();
	}

	/**
	 * Helper for the message line representation in the file.
	 */
	private static class MessageLine implements MessageAccess {

		private final String id = UUID.randomUUID().toString();
		private final String topic;
		private final String data;
		private final boolean isEmpty;

		private Map<String, Object> dataMap;
		private Map<String, Object> headersMap;

		public MessageLine(String line) throws IOException {
			String message = line.trim();
			this.isEmpty = message.length() == 0;
			if (!this.isEmpty) {
				int separator = message.indexOf(' ');
				if (separator == -1) {
					throw new IOException("Could not find separator between topic and data in message");
				}
				this.topic = message.substring(0, separator);
				this.data = message.substring(separator).trim();
			} else {
				this.topic = null;
				this.data = null;
			}
		}
		private MessageLine(String topic, String data) {
			this.topic = topic;
			this.data = data;
			this.isEmpty = false;
		}

		public boolean isEmpty() {
			return isEmpty;
		}

		@Override
		public String getId() {
			return id;
		}

		@Override
		public String getMessage() {
			return data;
		}

		@Override
		public String getBrokerTopic() {
			return topic;
		}

		@Override
		public void acknowledge() {
			// not used
		}

		@Override
		public Map<String, Object> getDataMap() {
			if (this.dataMap == null) {
				populateMaps();
			}

			return this.dataMap;
		}

		@Override
		public Map<String, Object> getHeadersMap() {
			if (this.headersMap == null) {
				populateMaps();
			}

			return this.headersMap;
		}

		private void populateMaps() {
			Pair<Map<String, Object>, Map<String, Object>> maps = toStructuredMessage(this.data);
			this.dataMap = maps.left;
			this.headersMap = maps.right;
		}

		@Override
		public String toString() {
			return topic + ' ' + data + System.lineSeparator();
		}
	}
}
