package com.sap.cds.services.impl.outbox.persistence.collectors;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import org.apache.commons.lang3.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.CdsLockTimeoutException;
import com.sap.cds.Result;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.services.environment.CdsProperties.Outbox.Persistent;
import com.sap.cds.services.impl.outbox.Messages;
import com.sap.cds.services.impl.outbox.Messages_;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutbox;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.outbox.OutboxUtils;

/**
 * Outbox collector implementation for a specific outbox partition.
 */
public class PartitionCollector implements Runnable {

	private static final Logger LOG = LoggerFactory.getLogger(PartitionCollector.class);

	private final CdsRuntime runtime;
	private final PersistenceService db;
	private final OutboxService outboxService;
	private final int chunkSize;
	private final int partition;

	private final Object pauseMonitor = new Object();
	private final AtomicInteger pauseCount = new AtomicInteger(5);
	private volatile boolean pause = false; // flag to synchronize wakeup

	private final long maxPauseMillis;
	private final long emitTimeoutSeconds;
	private final int maxPublishAttempts;
	private final boolean storeLastError;

	private final Supplier<List<TenantInfo>> tenantSupplier;

	public PartitionCollector(CdsRuntime runtime, PersistentOutbox outboxService,
			Supplier<List<TenantInfo>> tenantSupplier, int partition) {

		this.runtime = runtime;
		this.db = runtime.getServiceCatalog().getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
		this.outboxService = outboxService;
		this.partition = partition;

		Persistent persistent = runtime.getEnvironment().getCdsProperties().getOutbox().getPersistent();
		this.chunkSize = persistent.getChunkSize();
		this.maxPauseMillis = persistent.getMaxPause().getSeconds() * 1000;
		this.emitTimeoutSeconds = persistent.getEmitTimeout().getSeconds();
		this.maxPublishAttempts = persistent.getMaxAttempts();
		this.storeLastError = persistent.getStoreLastError().isEnabled();

		this.tenantSupplier = tenantSupplier;

		if (!storeLastError) {
			LOG.debug("Storing errors to the outbox is disabled.");
		}
	}

	@Override
	public void run() {
		processPartition();
	}

	private void pause() {
		synchronized(pauseMonitor) {
			pause = true;
			try {
				long pauseInMillis = getPauseMillis(pauseCount.get(), maxPauseMillis);
				LOG.debug("Pausing partition collector {} for {} ms", partition, pauseInMillis);
				pauseMonitor.wait(pauseInMillis);
			} catch (InterruptedException e) {
				LOG.debug("Partition collector thread '{}' interrupted", Thread.currentThread().getName());
				Thread.currentThread().interrupt();
			}
			pause = false;
		}
	}

	public void unpause() {
		// ensures that the next pause is short
		pauseCount.set(0);
		// wakes up a currently sleeping collector
		if (pause) {
			synchronized (pauseMonitor) {
				if(pause) {
					pause = false;
					pauseMonitor.notifyAll();
					LOG.debug("Notified paused partition collector {}", partition);
				}
			}
		}
	}

	private void processPartition() {
		while (!Thread.currentThread().isInterrupted()) {
			try {
				LOG.debug("Executing partition collector {}", partition);
				AtomicBoolean doPause = new AtomicBoolean(true);
				for (TenantInfo tenantInfo : this.tenantSupplier.get()) {
					try {
						LOG.debug("Processing tenant '{}' in partition collector {}", tenantInfo.getTenant(), partition);
						runtime.requestContext().clearUser().modifyUser(user -> user.setTenant(tenantInfo.getTenant())).run(req -> {
							if (OutboxUtils.hasOutboxModel(req.getModel())) {
								runtime.changeSetContext().run(ctx -> {
									CqnSelect select = Select.from(Messages_.class)
											.where(e -> e.partition().eq(partition)
													.and(e.attempts().lt(this.maxPublishAttempts)))
											.orderBy(e -> e.timestamp().asc())
											.limit(this.chunkSize)
											.lock(0);

									Result res = db.run(select);
									// at least one tenant still has more messages to process
									if(res.rowCount() >= select.top()) {
										doPause.set(false);
									}

									if(res.rowCount() > 0) {
										// track start of dispatch to interrupt if sequential dispatching takes too long
										final Instant startOfDispatch = Instant.now();
										for (Messages msg : res.listOf(Messages.class)) {
											// sequential publishing
											PublishState state = publish(msg, startOfDispatch);

											if (state == PublishState.SUCCESS) {
												db.run(Delete.from(Messages_.class).where(e -> e.ID().eq(msg.getId())));
											} else if (state == PublishState.TIMEOUT) {
												break;
											}

											// track time of dispatch process and interrupt if threshold has been reached
											if (Duration.between(startOfDispatch, Instant.now()).getSeconds() > this.emitTimeoutSeconds) {
												break;
											}
										}
									}
								});
							} else {
								LOG.debug("The outbox model is not available for the tenant '{}'", tenantInfo.getTenant());
							}
						});
					} catch (Exception e) {
						if (isLockTimeoutException(e)) {
							LOG.debug("Partition collector {} timed out waiting for table lock for tenant '{}'", partition, tenantInfo.getTenant());
							doPause.set(true);
							break;
						}
						LOG.warn("Exception occurred for tenant '{}' in partition collector {}", tenantInfo.getTenant(), partition, e);
					}
				}

				if(doPause.get()) {
					pause();
					if(pauseCount.get() < 20) {
						pauseCount.addAndGet(2);
					}
				} else {
					pauseCount.set(0);
				}
			} catch (Throwable e) {
				LOG.warn("Unexpected exception occured in partition collector {}", partition, e);
			}
		}
	}

	private PublishState publish(final Messages msg, final Instant startOfDispatch) {
		// by publishing a retry message we have to check whether the retry pause is reached before trying
		if (msg.getAttempts() != 0 && msg.getLastAttemptTimestamp() != null && (Duration.between(msg.getLastAttemptTimestamp(), Instant.now()).toMillis() < getPauseMillis(msg.getAttempts(), Integer.MAX_VALUE))) {
			LOG.debug("Message '{}' cannot be republished until the retry waiting time is reached", msg.getId());
			return PublishState.TIMEOUT;
		}

		LOG.debug("Publishing outbox message with target event '{}'", msg.getTarget());
		OutboxMessageEventContext ctx = OutboxMessageEventContext.create(msg.getTarget());
		ctx.setIsInbound(true);
		ctx.setTimestamp(msg.getTimestamp());
		ctx.setMessage(msg.getMsg());

		// recover the boxed context
		while (true) {
			try {
				outboxService.emit(ctx);
				return PublishState.SUCCESS;
			} catch (Throwable e) { // NOSONAR
				// we should may be check the CdsErrorStatuses.NO_ON_HANDLER exception in order
				// to ignore the entry without handling it as an error.
				LOG.warn("Failed to emit Outbox message with id '{}' and target '{}'", msg.getId(), msg.getTarget(), e);

				int currentAttempts = msg.getAttempts();

				// re-attempt to publish
				msg.setAttempts(++currentAttempts);
				msg.setLastAttemptTimestamp(Instant.now());
				Map<String, Object> data = new HashMap<>();
				data.put(Messages.ATTEMPTS, msg.getAttempts());
				data.put(Messages.LAST_ATTEMPT_TIMESTAMP, msg.getLastAttemptTimestamp());

				if (storeLastError) {
					StringWriter stringWriter = new StringWriter();
					e.printStackTrace(new PrintWriter(stringWriter));

					data.put(Messages.LAST_ERROR, stringWriter.toString());
				}

				db.run(Update.entity(Messages_.class).data(data).where(m -> m.ID().eq(msg.getId())));

				if (currentAttempts < this.maxPublishAttempts) {
					try {
						// exponential backoff in ms for re-attempt
						long pauseInMillis = getPauseMillis(currentAttempts, Integer.MAX_VALUE);

						// check time + pause of dispatch process and interrupt if threshold has been reached
						if (Duration.between(startOfDispatch, Instant.now().plusMillis(pauseInMillis)).getSeconds() > this.emitTimeoutSeconds) {
							LOG.debug("The retry waiting time of message '{}' would exceed the emit timeout, therefore release lock and commit transaction", msg.getId());
							return PublishState.TIMEOUT;
						}
						// wait till next try
						TimeUnit.MILLISECONDS.sleep(pauseInMillis);
					} catch (InterruptedException ie) {
						Thread.currentThread().interrupt();
					}
				} else {
					// giving up to publish
					LOG.warn("Reached maximum number of attempts to emit Outbox message with id '{}' and target '{}'",
							msg.getId(), msg.getTarget());
					return PublishState.FAILED;
				}
			}
		}
	}

	private static long getPauseMillis(int pauseCount, long maxTimeoutMillis) {
		long retryInMillis = Math.round(Math.pow(2d, pauseCount) * 1000 + RandomUtils.nextLong(0, 1001));
		return Math.min(retryInMillis, maxTimeoutMillis);
	}

	private static boolean isLockTimeoutException(Throwable t) {
		while (t != null) {
			if (t instanceof CdsLockTimeoutException) {
				return true;
			}
			t = t.getCause();
		}
		return false;
	}

	private static enum PublishState {
		SUCCESS,
		FAILED,
		TIMEOUT
	}
}