package com.sap.cds.services.impl.utils;

import java.util.Optional;

import io.opentelemetry.api.trace.SpanBuilder;
import io.opentelemetry.api.trace.SpanId;
import io.opentelemetry.api.trace.TraceId;
import io.opentelemetry.context.Context;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.services.impl.changeset.ChangeSetContextImpl;
import com.sap.cds.services.impl.request.RequestContextSPI;

import io.opentelemetry.api.GlobalOpenTelemetry;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.StatusCode;
import io.opentelemetry.api.trace.Tracer;
import io.opentelemetry.api.trace.TracerProvider;
import org.slf4j.event.Level;

public class OpenTelemetryUtils {

	private final static Logger logger = LoggerFactory.getLogger(OpenTelemetryUtils.class);

	private final static String CDS_INSTRUMENTATION_SCOPE = "com.sap.cds";
	private static Tracer tracer = GlobalOpenTelemetry.get().getTracer(CDS_INSTRUMENTATION_SCOPE);

	public enum CdsSpanType {
		REQUEST_CONTEXT (Level.INFO),
		CHANGESET_CONTEXT (Level.INFO),
		EMIT (Level.DEBUG);

		private final Level logLevel;

		CdsSpanType(Level logLevel) {
			this.logLevel = logLevel;
		}

		public Level getLogLevel() {
			return this.logLevel;
		}
	}

	private OpenTelemetryUtils() {
		// hidden
	}

	public static Optional<Span> createSpan(CdsSpanType type) {
		return createSpan(type, null);
	}

	public static Optional<Span> createSpan(CdsSpanType type, Context parentContext) {
		if(logger.isEnabledForLevel(type.getLogLevel())) {
			SpanBuilder spanBuilder =  tracer.spanBuilder(type.toString());

			/*
			 * For Java Executors, auto instrumentation might be in place. So, if there is a valid trace
			 * context available, we treat it with precedence over the manual propagation and will be
			 * automatically propagated. No need for us to set the parent manually.
			 */
			if(!isValidContext(Context.current()) && parentContext != null) {
				spanBuilder.setParent(parentContext);
			}

			return Optional.of(spanBuilder.startSpan());
		}
		return Optional.empty();
	}

	private static boolean isValidContext(Context ctx) {
		Span span = Span.fromContext(ctx);
		return TraceId.isValid(span.getSpanContext().getTraceId()) && SpanId.isValid(span.getSpanContext().getSpanId());
	}
  
	public static void updateSpan(Optional<Span> span, String serviceName, String eventName, String entityName) {
		span.ifPresent(s -> {
			s.updateName(serviceName + " (" + eventName + ")");
			s.setAttribute("cds.service", String.valueOf(serviceName));
			s.setAttribute("cds.eventName", String.valueOf(eventName));
			s.setAttribute("cds.entityName", String.valueOf(entityName));
		});
	}

	public static void updateSpan(Optional<Span> span, RequestContextSPI requestContext) {
		span.ifPresent(s -> {
			s.updateName("RequestContext " + requestContext.getId());
			s.setAttribute("cds.tenant", String.valueOf(requestContext.getUserInfo().getTenant()));
			s.setAttribute("cds.systemUser", String.valueOf(requestContext.getUserInfo().isSystemUser()));
			s.setAttribute("cds.internalUser", String.valueOf(requestContext.getUserInfo().isInternalUser()));
			s.setAttribute("cds.privilegedUser", String.valueOf(requestContext.getUserInfo().isPrivileged()));
			s.setAttribute("cds.locale", String.valueOf(requestContext.getParameterInfo().getLocale()));
			s.setAttribute("cds.validFrom", String.valueOf(requestContext.getParameterInfo().getValidFrom()));
			s.setAttribute("cds.validTo", String.valueOf(requestContext.getParameterInfo().getValidTo()));
		});
	}

	public static void updateSpan(Optional<Span> span, ChangeSetContextImpl changeSetContext) {
		span.ifPresent(s -> {
			s.updateName("ChangeSetContext " + changeSetContext.getId());
		});
	}

	public static void endSpan(Optional<Span> span) {
		span.ifPresent(Span::end);
	}

	public static void recordException(Optional<Span> span, Exception e) {
		span.ifPresent(s -> {
			s.recordException(e);
			s.setStatus(StatusCode.ERROR);
		});
	}

	@VisibleForTesting
	public static void setTracerProvider(TracerProvider tracerProvider) {
		tracer = tracerProvider.tracerBuilder(CDS_INSTRUMENTATION_SCOPE).setInstrumentationVersion("1.0.0").build();
	}
}
