/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Expand;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.impl.utils.TargetAwareCqnModifier;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;

class PersonalDataModifier extends TargetAwareCqnModifier {

	private final List<PersonalDataModifier> children = new LinkedList<>();
	private boolean innerCqn;
	private final boolean addPersonalData;
	private final PersonalDataCaches caches;

	/**
	 * Stores the accessed sensitive elements of all accessed entities
	 */
	private final Map<String, Pair<CdsStructuredType, List<CqnElementRef>>> sensitiveElements = new HashMap<>();

	/**
	 *
	 * @param target          the target entity
	 * @param addPersonalData if {@code true}, also elements annotated with {@code @PersonalData.IsPotentiallyPersonal} are considered
	 * @param caches          caches for personal data meta and utils
	 */
	PersonalDataModifier(CdsEntity target, boolean addPersonalData, PersonalDataCaches caches) {
		this(target, false, addPersonalData, caches);
	}

	private PersonalDataModifier(CdsEntity target, boolean innerCqn, boolean addPersonalData, PersonalDataCaches caches) {
		super(Objects.requireNonNull(target, "target must not null"));
		this.innerCqn = innerCqn;
		this.addPersonalData = addPersonalData;
		this.caches = Objects.requireNonNull(caches, "caches must not null");
	}

	@Override
	public CqnStructuredTypeRef ref(CqnStructuredTypeRef ref) {
		// we have reached inner cqn statement here.
		// ref() is guaranteed to be called before where() within a statement.
		this.innerCqn = true;
		return ref;
	}

	@Override
	protected TargetAwareCqnModifier create(CdsEntity target) {
		PersonalDataModifier result = new PersonalDataModifier(target, this.innerCqn, this.addPersonalData, this.caches);
		this.children.add(result);
		return result;
	}

	@Override
	public List<CqnSelectListItem> selectAll() {
		// adds sensitive fields to selected items
		getTarget().elements().filter(element -> element.getType().isSimple() || element.getType().isArrayed())
				.forEach(element -> addIfSensitive(element, CQL.get(element.getName())));
		return super.selectAll();
	}

	@Override
	public CqnValue ref(CqnElementRef ref) {
		if (this.innerCqn) {
			CdsModelUtils.findElement(getTarget(), ref).ifPresent(element -> addIfSensitive(element, ref));
		}
		return ref;
	}

	private void addIfSensitive(CdsElement element, CqnElementRef ref) {
		if (CdsAnnotations.PERSONALDATA_POTENTIALLYSENSITIVE.isTrue(element)
				|| (CdsAnnotations.PERSONALDATA_POTENTIALLYPERSONAL.isTrue(element) && this.addPersonalData)) {
			// select keys and data subject id
			CdsStructuredType def = element.getDeclaringType();
			String key = def.getQualifiedName();

			if (this.sensitiveElements.containsKey(key)) {
				List<CqnElementRef> elementList = this.sensitiveElements.get(key).getRight();
				// avoid duplicate entries in sensitiveElements
				Optional<CqnElementRef> elementRefOpt = elementList.stream()
						.filter(elementRef -> elementRef.lastSegment().equals(ref.lastSegment())).findFirst();
				if (elementRefOpt.isEmpty()) {
					this.sensitiveElements.get(key).getRight().add(ref);
				}
			} else {
				this.sensitiveElements.put(key, Pair.of(def, new LinkedList<>(Arrays.asList(ref))));
			}
		}
	}

	@Override
	public List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
		if (this.innerCqn) {
			this.sensitiveElements.entrySet().forEach(entry -> {
				// resolve the keys and data subject id for the accessed entity
				CqnElementRef ref = entry.getValue().getRight().get(0);

				LinkedList<Segment> segments = new LinkedList<>(ref.segments());
				segments.removeLast();
				PersonalDataMeta pdMeta = getMeta(entry.getValue().getLeft());

				// TODO check if key is already in select list and don't add it again
				Set<CqnSelectListValue> elements = pdMeta.getKeyNames().stream()
						.map(str -> CQL.get(str).as(PersonalDataUtils.getKeyAlias(str))).collect(Collectors.toSet());

				elements.addAll(getUtils(entry.getValue().getLeft()).resolveDataSubjectId());
				if (segments.isEmpty()) {
					elements.forEach(items::add);
				} else {
					// for refs with more than one segment, expand all segments
					// TODO add alias to expand
					items.add(expandPath(segments, elements));
				}
			});
			this.innerCqn = false;  // ignore refs in groupBy/orderBy
		}
		return super.items(items);
	}

	private Expand<?> expandPath(LinkedList<Segment> segments, Set<CqnSelectListValue> items) {
		if (segments.size() == 1) {
			return CQL.to(segments).expand(items.toArray(new CqnSelectListValue[0]));
		}
		Segment segment = segments.removeFirst();
		return CQL.to(Arrays.asList(segment)).expand(expandPath(segments, items));
	}

	Map<String, Pair<CdsStructuredType, List<CqnElementRef>>> getSensitiveElements() {
		Map<String, Pair<CdsStructuredType, List<CqnElementRef>>> result = new HashMap<>(this.sensitiveElements);
		// traverse into all children
		this.children.forEach(child -> {
			Map<String, Pair<CdsStructuredType, List<CqnElementRef>>> sensitiveElementsChild = child.getSensitiveElements();
			sensitiveElementsChild.entrySet().forEach(entry -> {
				if (result.containsKey(entry.getKey())) {
					result.get(entry.getKey()).getRight().addAll(entry.getValue().getRight());
				} else {
					result.put(entry.getKey(), entry.getValue());
				}
			});
		});
		return result;
	}

	private PersonalDataUtils getUtils(CdsStructuredType entity) {
		return this.caches.getUtils(entity);
	}

	private PersonalDataMeta getMeta(CdsStructuredType entity) {
		return this.caches.getMeta(entity);
	}
}
