/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.DELETABLE;
import static com.sap.cds.services.utils.model.CdsAnnotations.INSERTABLE;
import static com.sap.cds.services.utils.model.CdsAnnotations.INSERTONLY;
import static com.sap.cds.services.utils.model.CdsAnnotations.READABLE;
import static com.sap.cds.services.utils.model.CdsAnnotations.READONLY;
import static com.sap.cds.services.utils.model.CdsAnnotations.UPDATABLE;

import java.util.Iterator;

import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.model.CdsModelUtils;

@ServiceName(value = "*", type = ApplicationService.class)
public class CapabilitiesHandler implements EventHandler {

	// default handlers
	@Before(event = CqnService.EVENT_READ)
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityRead(EventContext context) {
		if(!getCapabilities(context).isReadable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_READABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityCreate(EventContext context) {
		if(!getCapabilities(context).isInsertable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_INSERTABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = { CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityUpdate(EventContext context) {
		if(!getCapabilities(context).isUpdatable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_UPDATABLE, context.getTarget().getQualifiedName());
		}
	}

	@Before(event = CqnService.EVENT_DELETE)
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	public void checkCapabilityDelete(EventContext context) {
		if(!getCapabilities(context).isDeletable()) {
			throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_DELETABLE, context.getTarget().getQualifiedName());
		}
	}

	/**
	 * Returns the {@link Capabilities} for a {@link CdsEntity}.
	 * The {@link Capabilities} describe CRUD restrictions on the entity.
	 *
	 * Annotations {@literal @readonly} and {@literal @insertonly} take precedence before
	 * {@literal @Capabilities.Insertable}, {@literal @Capabilities.Updatable}, {@literal @Capabilities.Deletable}
	 *
	 * @param context the {@link EventContext} whose target {@link CdsEntity} should be checked
	 * @return the {@link Capabilities}
	 */
	public static Capabilities getCapabilities(EventContext context) {
		CdsEntity entity = context.getTarget();
		CqnStatement cqn = (CqnStatement) context.get("cqn");
		// auto-exposed composition entities are generally not accessible
		if(isInvalidAutoExposedRootEntity(cqn, context.getModel())) {
			return new Capabilities(false, false, false, false);
		}

		// auto-exposed value-helps and their childs are generally readonly
		if(isExplicitlyAutoExposedTargetEntity(cqn, context.getModel())) {
			return new Capabilities(true, false, false, false);
		}

		// readonly and insertonly take precedence
		boolean readonly = READONLY.isTrue(entity);
		if(readonly) {
			return new Capabilities(true, false, false, false);
		}

		boolean insertOnly = INSERTONLY.isTrue(entity);
		if(insertOnly) {
			return new Capabilities(false, true, false, false);
		}

		// capabilities default to true
		boolean insertable = defaultToTrue(INSERTABLE.getOrDefault(entity));
		boolean updatable = defaultToTrue(UPDATABLE.getOrDefault(entity));
		boolean deletable = defaultToTrue(DELETABLE.getOrDefault(entity));
		boolean readable = defaultToTrue(READABLE.getOrDefault(entity));

		return new Capabilities(readable, insertable, updatable, deletable);
	}

	/**
	 * This method ensures, that the capability is only set to false, if "false" was provided in the annotation.
	 * True or any other string are treated as true.
	 * @param value
	 * @return
	 */
	private static boolean defaultToTrue(Object value) {
		if(value instanceof Boolean boolean1) {
			return boolean1;
		} else if (value instanceof String string) {
			return !string.equalsIgnoreCase("false");
		}

		return true;
	}

	/**
	 * Checks if the CQN statement's root entity is an invalid root entity.
	 * In general auto-exposed composition entities are an invalid root entity of a path.
	 * The path of a statement should always start in an explicitly exposed entity,
	 * or an entity that was explicitly wished to be auto-exposed (@cds.autoexpose).
	 *
	 * Draft-enabled entities are an exception to this rule,
	 * as Fiori always expects these entities to be available directly.
	 *
	 * @param cqn the cqn statement, whose ref is analyzed
	 * @param model the {@link CdsModel}
	 * @return true, if the CQN statement's root entity is an invalid root entity.
	 */
	private static boolean isInvalidAutoExposedRootEntity(CqnStatement cqn, CdsModel model) {
		if (cqn.isSelect() && cqn.asSelect().from().isSelect()) {
			return isInvalidAutoExposedRootEntity(cqn.asSelect().from().asSelect(), model);
		}
		CdsEntity rootEntity = CdsModelUtils.getEntityPath(cqn.ref(), model).rootEntity();
		boolean autoexposed = CdsAnnotations.AUTOEXPOSED.isTrue(rootEntity);
		boolean autoexpose = CdsAnnotations.AUTOEXPOSE.isTrue(rootEntity);
		return autoexposed && !autoexpose && !DraftUtils.isDraftEnabled(rootEntity);
	}

	/**
	 * Checks if the CQN statement's path is targeting an entity that was explicitly wished to be auto-exposed (@cds.autoexpose)
	 * or one of its childs, which is considered explicitly wished to be auto-exposed as well.
	 *
	 * Explicitly wished to be auto-exposed entities are usually value helps.
	 * Most importantly composition entities should generally not be wished to be auto-exposed explicitly (@cds.autoexpose),
	 * as they are already implicitly auto-exposed through their composition's parent exposure.
	 *
	 * The _texts entities generated by the compiler are erronously annotated with @cds.autoexpose,
	 * even though they have a composition relation to the entity defining the localized elements.
	 * They are therefore explicitly excluded in this logic.
	 *
	 * @param cqn the cqn statement, whose ref is analyzed
	 * @param model the {@link CdsModel}
	 * @return true, if the CQN statement's path is targeting an entity that was explicitly wished to be auto-exposed
	 */
	private static boolean isExplicitlyAutoExposedTargetEntity(CqnStatement cqn, CdsModel model) {
		if (cqn.isSelect() && cqn.asSelect().from().isSelect()) {
			return isExplicitlyAutoExposedTargetEntity(cqn.asSelect().from().asSelect(), model);
		}
		Iterator<ResolvedSegment> reverse = CdsModelUtils.getEntityPath(cqn.ref(), model).reverse();
		while(reverse.hasNext()) {
			CdsEntity entity = reverse.next().entity();
			boolean autoexposed = CdsAnnotations.AUTOEXPOSED.isTrue(entity);
			boolean autoexpose = CdsAnnotations.AUTOEXPOSE.isTrue(entity);
			// while walking the statement's ref backwards we found a non autoexposed entity
			// we abort, as this statement apparently does not target an explicitly auto-exposed entity.
			if (!autoexposed) {
				break;
			}
			// the entity is explicitly wished to be auto-exposed (@cds.autoexpose)
			// and in addition was in fact auto-exposed
			// _texts entities are erronously annotated with @cds.autoexpose,
			// as they are exposed through composition relation anyway already
			else if(autoexpose && !entity.getQualifiedName().endsWith("_texts") && !entity.getQualifiedName().endsWith(".texts")) {
				return true;
			}

		}
		return false;
	}

	public static class Capabilities {

		private boolean readable;
		private boolean insertable;
		private boolean updatable;
		private boolean deletable;

		private Capabilities(boolean readable, boolean insertable, boolean updatable, boolean deletable) {
			this.readable = readable;
			this.insertable = insertable;
			this.updatable = updatable;
			this.deletable = deletable;
		}

		public boolean isReadable() {
			return readable;
		}

		public boolean isInsertable() {
			return insertable;
		}

		public boolean isUpdatable() {
			return updatable;
		}

		public boolean isDeletable() {
			return deletable;
		}

	}

}
