/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.ql.cqn.transformation.CqnGroupByTransformation;
import com.sap.cds.ql.cqn.transformation.CqnTransformation;
import com.sap.cds.ql.impl.ExpressionVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.TargetAwareCqnModifier;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

@ServiceName(value = "*", type = ApplicationService.class)
public class ImplicitSortingHandler implements EventHandler {

	@Before
	@HandlerOrder(OrderConstants.Before.FILTER_FIELDS)
	public void implicitSorting(CdsReadEventContext context) {
		CqnSelect select = context.getCqn();
		if (select.from().isRef()) {
			List<CqnValue> groupBy = new ArrayList<>();
			if (!select.groupBy().isEmpty()) {
				// grouped by group by clause
				groupBy.addAll(select.groupBy());
			} else if (!select.transformations().isEmpty()) {
				// possibly grouped by transformation (last one wins)
				groupBy.addAll(lastGroupByTransformation(select.transformations()));
			}
			boolean onlyFunctions = selectsFunctionsOrLiteralsOnly(select.items()) || hasAggregateTransformation(select);
			CqnSelect copy = CQL.copy(select, new OrderByModifier(context.getTarget(), onlyFunctions, groupBy, select.hasLimit()));
			context.setCqn(copy);
		}
	}

	private boolean hasAggregateTransformation(CqnSelect select) {
		List<CqnTransformation> trafos = select.transformations();
		ListIterator<CqnTransformation> iter = trafos.listIterator(trafos.size());
		while(iter.hasPrevious()) {
			CqnTransformation trafo = iter.previous();
			if (trafo.kind() == CqnTransformation.Kind.AGGREGATE) {
				return true;
			}
		}
		
		return false;
	}

	private List<CqnElementRef> lastGroupByTransformation(List<CqnTransformation> trafos) {
		ListIterator<CqnTransformation> iter = trafos.listIterator(trafos.size());
		while(iter.hasPrevious()) {
			CqnTransformation trafo = iter.previous();
			if (trafo.kind() == CqnTransformation.Kind.GROUPBY) {
				return ((CqnGroupByTransformation) trafo).dimensions();
			}
		}
		return List.of();
	}

	public static class OrderByModifier extends TargetAwareCqnModifier {

		private boolean mainOnlyFunctions = false;
		private List<CqnValue> mainGroupBy = List.of();
		private boolean mainLimit = false;

		public OrderByModifier(CdsEntity target, boolean mainOnlyFunctions, List<CqnValue> mainGroupBy, boolean mainLimit) {
			super(target);
			this.mainOnlyFunctions = mainOnlyFunctions;
			this.mainGroupBy = mainGroupBy;
			this.mainLimit = mainLimit;
		}

		@Override
		protected TargetAwareCqnModifier create(CdsEntity target) {
			return new OrderByModifier(target, false, List.of(), false);
		}

		@Override
		public List<CqnSortSpecification> orderBy(List<CqnSortSpecification> orderBy) {
			return extendOrderBy(getTarget(), orderBy, mainOnlyFunctions, mainGroupBy, mainLimit);
		}

		@Override
		public CqnSelectListItem expand(CqnExpand expand) {
			CdsEntity refTarget = getRefTarget(expand.ref());
			if (refTarget != null) {
				List<CqnSortSpecification> orderBy = extendOrderBy(refTarget, new ArrayList<>(expand.orderBy()), selectsFunctionsOrLiteralsOnly(expand.items()), Collections.emptyList(), expand.hasLimit());
				return CQL.copy(expand).orderBy(orderBy);
			}
			return expand;
		}

		private List<CqnSortSpecification> extendOrderBy(CdsEntity target, List<CqnSortSpecification> orderBy, boolean onlyFunctions, List<CqnValue> groupBy, boolean limit) {
			// in case of functions we assume the functions are aggregating and the result only contains a single row
			// therefore we don't add any sorting specifications
			// this might not always be the case, but for now we ignore these edge cases
			if (onlyFunctions) {
				return orderBy;
			}

			Set<String> sortItems = new HashSet<>();
			orderBy.stream().map(si -> si.value().toJson()).forEach(sortItems::add);

			// apply explicitly modeled sorting specifications
			// if query is structurally equivalent to the entity
			if (groupBy.isEmpty()) {
				// ordering defined in the @cds.default.order annotation
				Object defaultOrder = CdsAnnotations.DEFAULT_ORDER.getOrDefault(target);
				handleDefaultOrderBy(defaultOrder, orderBy);

				// ordering defined in the underlying view
				// if we find an empty 'order by' on the underlying view we don't analyze the view's view
				target.query().ifPresent(view -> view.orderBy().stream().forEach(o -> {
					String json = o.value().toJson();
					if (!sortItems.contains(json)) {
						sortItems.add(o.value().toJson());
						orderBy.add(ExpressionVisitor.copy(o, new Modifier() {
						}));
					}
				}));
			}

			// implicit sorting to ensure stable paging
			if (limit) {
				if (groupBy.isEmpty()) {
					// sort by keys
					// TODO: Should we ensure a stable order of the keys?
					target.keyElements()
							.filter(e -> e.getType().isSimple())
							.filter(e -> !contains(e.getName(), orderBy))
							.map(e -> CQL.get(e.getName()).asc())
							.forEach(orderBy::add);
				} else {
					// sort by all group by elements that are not already included in the orderBy explicitly
					groupBy.stream()
							.flatMap(CqnValue::ofRef)
					  		.filter(e -> !contains(e.path(), orderBy))
							.map(v -> CQL.get(v.segments()).asc())
							.forEach(orderBy::add);
				}
			}
			return orderBy;
		}

		private void handleDefaultOrderBy(Object value, List<CqnSortSpecification> orderBy) {
			if (value instanceof Map<?, ?> map) {
				Object by = map.get("by");
				Object element = map.get("=");
				if(by instanceof Map<?, ?> map1) {
					Object innerElement = map1.get("=");
					if(innerElement instanceof String string && !contains(string, orderBy)) {
						CqnValue val = CQL.get(string);
						Object desc = map.get("desc");
						CqnSortSpecification.Order order = CqnSortSpecification.Order.ASC;
						if(Boolean.TRUE.equals(desc)) {
							order = CqnSortSpecification.Order.DESC;
						}
						orderBy.add(CQL.sort(val, order));
					}
				} else if (element instanceof String string && !contains(string, orderBy)) {
					orderBy.add(CQL.get(string).asc());
				}
			} else if (value instanceof List<?> list) {
				list.forEach(o -> handleDefaultOrderBy(o, orderBy));
			}
		}

		private boolean contains(String element, List<CqnSortSpecification> orderBy) {
			return orderBy.stream().anyMatch(c -> refNamed(c.value(), element));
		}

		private boolean refNamed(CqnValue val, String name) {
			if (!val.isRef()) {
				return false;
			}

			return name.equals(val.asRef().path());
		}

	}
	
	private static boolean selectsFunctionsOrLiteralsOnly(List<CqnSelectListItem> items) {
		return !items.isEmpty() && items.stream().allMatch(item -> item.isValue()
				&& (item.asValue().value().isFunction() || item.asValue().value().isLiteral()));
	}

}
