/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.sap.cds.Struct;
import com.sap.cds.impl.ProxyList;
import com.sap.cds.ql.CdsName;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.utils.StringUtils;

public class TypedCqnServiceInvocationHandler implements InvocationHandler {

	private final CqnService service;
	private final Map<Method, MethodInfo> methodInfos = new HashMap<>();

	private record MethodInfo(String event, String entity, String[] argNames) {
	}

	TypedCqnServiceInvocationHandler(CqnService service, Class<? extends CqnService> generatedInterface) {
		this.service = service;
		// analyze methods of generated interface
		for (Method method : generatedInterface.getDeclaredMethods()) {
			String eventName = getCdsName(method).orElse(method.getName());
			String entityName = null;
			Parameter[] parameters = method.getParameters();
			String[] argNames = new String[parameters.length];
			for (int i = 0; i < parameters.length; ++i) {
				Parameter parameter = parameters[i];
				if (i == 0 && StructuredType.class.isAssignableFrom(parameter.getType())) {
					argNames[i] = "cqn";
					entityName = getCdsName(parameter.getType()).orElse(null);
				} else {
					argNames[i] = getCdsName(parameter).orElse(parameter.getName());
				}
			}
			methodInfos.put(method, new MethodInfo(eventName, entityName, argNames));
		}
	}

	@Override
	public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
		MethodInfo methodInfo = methodInfos.get(method);
		// emit event
		if (methodInfo != null) {
			EventContext eventContext = createEventContext(methodInfo, args);
			service.emit(eventContext);
			Object result = eventContext.get("result");
			return transformResult(result, method);
		}
		// invoke original method
		try {
			return method.invoke(service, args);
		} catch (InvocationTargetException e) {
			throw e.getTargetException();
		}
	}

	public Service getDelegatedService() {
		return service;
	}

	private static EventContext createEventContext(MethodInfo methodInfo, Object[] args) {
		EventContext eventContext = EventContext.create(methodInfo.event, methodInfo.entity);
		if (args != null) {
			for (int i = 0; i < args.length; ++i) {
				Object arg = args[i];
				// if first argument is a structured type, it's the ref for a bound action / function
				// and it needs to be converted to a CqnSelect
				if (i == 0 && arg instanceof StructuredType<?> structuredType) {
					arg = Select.from(structuredType);
				}
				eventContext.put(methodInfo.argNames[i], arg);
			}
		}
		return eventContext;
	}

	static Optional<String> getCdsName(AnnotatedElement annotated) {
		CdsName cdsName = annotated.getAnnotation(CdsName.class);
		if (cdsName != null && !StringUtils.isEmpty(cdsName.value())) {
			return Optional.of(cdsName.value());
		}
		return Optional.empty();
	}

	@SuppressWarnings({"rawtypes", "unchecked" })
	private static Object transformResult(Object result, Method method) {
		if (result == null) {
			return result;
		}

		Class<?> returnType = method.getReturnType();
		if (Map.class.isAssignableFrom(returnType)) {
			if (result.getClass().isAssignableFrom(returnType) || Map.class == returnType) {
				return result;
			}
			return Struct.access((Map<String, Object>) result).as(returnType);
		}
		if (Collection.class.isAssignableFrom(returnType)) {
			Type genericReturnType = method.getGenericReturnType();
			if (genericReturnType instanceof ParameterizedType parameterizedType) {
				Type type = parameterizedType.getActualTypeArguments()[0];
				if (type instanceof Class itemClass && Map.class.isAssignableFrom(itemClass) && Map.class != itemClass) {
					return new ProxyList((List<Map<String, Object>>) result, itemClass);
				} else {
					return result;
				}
			}
			throw new IllegalStateException(
					"Return type of service interface method " + method.getName() + " must be parameterized");
		}
		return result;
	}
}
