/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.environment.CdsProperties.Outbox;
import com.sap.cds.services.environment.CdsProperties.Outbox.OutboxServiceConfig;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutbox;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutboxInitializationHandler;
import com.sap.cds.services.impl.outbox.persistence.collectors.TenantCache;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.outbox.OutboxUtils;

public class OutboxServiceConfiguration implements CdsRuntimeConfiguration {

	private static final Logger logger = LoggerFactory.getLogger(OutboxServiceConfiguration.class);

	private final TenantCache tenantCache = new TenantCache();

	@Override
	@SuppressWarnings("deprecation")
	public void services(CdsRuntimeConfigurer configurer) {
		CdsRuntime runtime = configurer.getCdsRuntime();
		if (OutboxUtils.hasOutboxModel(runtime.getCdsModel())) {
			runtime.getEnvironment().getCdsProperties().getOutbox().getServices().values().forEach(config -> {
				boolean ordered = getOrderedFlag(config);
				configurer.service(new PersistentOutbox(config.getName(), config, ordered, runtime, tenantCache::getTenants));
			});

			Outbox.Persistent defaultConfig = runtime.getEnvironment().getCdsProperties().getOutbox().getPersistent();
			if (defaultConfig.isEnabled()) {
				if (runtime.getServiceCatalog().getService(OutboxService.class, OutboxService.PERSISTENT_ORDERED_NAME) == null) {
					configurer.service(new PersistentOutbox(OutboxService.PERSISTENT_ORDERED_NAME, defaultConfig, true, runtime, tenantCache::getTenants));
				}

				if (runtime.getServiceCatalog().getService(OutboxService.class, OutboxService.PERSISTENT_UNORDERED_NAME) == null) {
					configurer.service(new PersistentOutbox(OutboxService.PERSISTENT_UNORDERED_NAME, defaultConfig, false, runtime, tenantCache::getTenants));
				}
			}
		} else if (!runtime.getEnvironment().getCdsProperties().getOutbox().getServices().isEmpty()) {
			logger.warn("Explicitly configured outbox services will not be available, because persistent outbox model is not available.");
		}

		if (runtime.getEnvironment().getCdsProperties().getOutbox().getInMemory().isEnabled()) {
			configurer.service(new InMemoryOutbox(OutboxService.INMEMORY_NAME, runtime));
		}
	}

	private boolean getOrderedFlag(OutboxServiceConfig config) {
		if (OutboxService.PERSISTENT_ORDERED_NAME.equals(config.getName())) {
			return true;
		} else if (OutboxService.PERSISTENT_UNORDERED_NAME.equals(config.getName())) {
			return false;
		}

		return config.isOrdered();
	}

	@Override
	public void eventHandlers(CdsRuntimeConfigurer configurer) {
		if (configurer.getCdsRuntime().getServiceCatalog().getServices(PersistentOutbox.class).findAny().isPresent()) {
			configurer.eventHandler(new PersistentOutboxInitializationHandler(tenantCache));
		}
		configurer.eventHandler(new CqnServiceOutboxHandler());
	}

	@Override
	public int order() {
		return -100; // before usual service configurations
	}

}
