/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox.persistence;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.impl.parser.token.Jsonizer;
import com.sap.cds.ql.Insert;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.changeset.ChangeSetListener;
import com.sap.cds.services.environment.CdsProperties.Outbox.OutboxConfig;
import com.sap.cds.services.impl.outbox.AbstractOutboxService;
import com.sap.cds.services.impl.outbox.Messages;
import com.sap.cds.services.impl.outbox.Messages_;
import com.sap.cds.services.impl.outbox.persistence.collectors.PartitionCollector;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;

public class PersistentOutbox extends AbstractOutboxService {

	private static final Logger LOG = LoggerFactory.getLogger(PersistentOutbox.class);
	public static final String ATTR_EVENT = "event";
	public static final String ATTR_MESSAGE = "message";

	private Thread job;
	private final PartitionCollector collector;
	private final OutboxConfig config;

	// in order to keep single registration for the same change set
	private Set<ChangeSetContext> changeSetContextCache = ConcurrentHashMap.newKeySet();

	public PersistentOutbox(String name, OutboxConfig config, boolean ordered, CdsRuntime runtime, Supplier<List<TenantInfo>> tenantSupplier) {
		super(name, runtime);
		this.config = config;
		this.collector = new PartitionCollector(runtime, this, config, ordered, tenantSupplier);
	}

	void init() {
		if (job == null) {
			LOG.debug("Initializing collector of outbox '{}'", getName());
			job = new Thread(collector, getName() + "-collector");
			job.setDaemon(true);
			job.start();
		}
	}

	void stop() {
		if (job != null) {
			LOG.debug("Stopping collector of the outbox '{}'", getName());
			// will interrupt collector
			job.interrupt();
		}
	}

	@Override
	protected void submit(OutboxMessageEventContext context) {
		LOG.debug("Submitting outbox message for target '{}' with event '{}'.", getName(), context.getEvent());
		persist(context);
		LOG.debug("Stored outbox message for target '{}' with event '{}'.", getName(), context.getEvent());

		ChangeSetContext changeSetContext = context.getChangeSetContext();
		if (config.getTriggerSchedule().isEnabled() && !changeSetContextCache.contains(changeSetContext)) {

			changeSetContextCache.add(changeSetContext);
			// schedule the outbox at the end of transaction
			changeSetContext.register(new ChangeSetListener() {
				@Override
				public void afterClose(boolean completed) {
					changeSetContextCache.remove(changeSetContext);
					if (completed) {
						collector.unpause();
					}
				}
			});
		}
	}

	private void persist(OutboxMessageEventContext context) {
		Messages message = Messages.create();
		// set the outbox target
		Map<String, Object> data = new HashMap<>();
		data.put(ATTR_MESSAGE, context.getMessage());
		data.put(ATTR_EVENT, context.getEvent());
		message.setMsg(Jsonizer.json(data));
		message.setTarget(getName());
		message.setTimestamp(context.getTimestamp());

		PersistenceService db = CdsServiceUtils.getDefaultPersistenceService(context);
		Insert insert = Insert.into(Messages_.class).entry(message);
		db.run(insert);
	}

}
