/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.stream.Stream;

import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.RemoteService;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.messaging.MessagingService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeAware;
import com.sap.cds.services.utils.info.CdsInfo;

public class ServiceCatalogInfo implements CdsInfo, CdsRuntimeAware {

	public static enum Info {

		TYPE("type", ServiceCatalogInfo::type, Details.LOW),

		PROPERTIES("properties", ServiceCatalogInfo::properties, Details.MEDIUM),

		HANDLERS("handlers", ServiceCatalogInfo::handlers, Details.HIGH);

		String name;
		BiFunction<Service, CdsRuntime, ?> valueFunc;
		Details minFlavor;

		private Info(String name, BiFunction<Service, CdsRuntime, ?> valueFunc, Details minFlavor) {
			this.name = name;
			this.valueFunc = valueFunc;
			this.minFlavor = minFlavor;
		}

		public String getName() {
			return name;
		}
	}

	private CdsRuntime runtime;

	@Override
	public String name() {
		return "services";
	}

	@Override
	public void setCdsRuntime(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public Map<String, Object> info(Details flavor) {

		Map<String, Object> result = new HashMap<>();

		if (runtime != null) {
			ServiceCatalog catalog = runtime.getServiceCatalog();
			catalog.getServices().forEach(service -> {
				Map<String, Object> serviceMap = new HashMap<>();

				Stream.of(Info.values()).filter(i -> flavor.compareTo(i.minFlavor) >= 0).forEach(info -> {
					Object value = info.valueFunc.apply(service, runtime);
					if (value != null) {
						serviceMap.put(info.name, value);
					}
				});

				result.put(service.getName(), serviceMap);
			});
		}

		return result;
	}

	private static String type(Service service, CdsRuntime runtime) {
		return CdsServiceUtils.getServiceType(service).getName();
	}

	private static Object properties(Service service, CdsRuntime runtime) {
		CdsProperties properties = runtime.getEnvironment().getCdsProperties();
		if(service instanceof ApplicationService) {
			return properties.getApplication().getServices().values().stream().filter(
				srv -> service.getName().equals(srv.getName())).findAny().orElse(null);
		} else if (service instanceof RemoteService) {
			return properties.getRemote().getServices().values().stream().filter(
				srv -> service.getName().equals(srv.getName())).findAny().orElse(null);
		} else if (service instanceof MessagingService) {
			return properties.getMessaging().getServices().values().stream().filter(
				srv -> service.getName().equals(srv.getName())).findAny().orElse(null);
		}
		return null;
	}

	private static Map<String, Object> handlers(Service service, CdsRuntime runtime) {
		ServiceSPI serviceSPI = CdsServiceUtils.getServiceSPI(service);
		if (serviceSPI != null) {
			Map<String, Object> handlers = new LinkedHashMap<>();

			List<HandlerDescription> beforeHandlers = createHandlerDescriptions(serviceSPI, Phase.BEFORE);
			if(!beforeHandlers.isEmpty()) {
				handlers.put(Before.class.getSimpleName(), beforeHandlers);
			}
			List<HandlerDescription> onHandlers = createHandlerDescriptions(serviceSPI, Phase.ON);
			if (!onHandlers.isEmpty()) {
				handlers.put(On.class.getSimpleName(), onHandlers);
			}
			List<HandlerDescription> afterHandlers = createHandlerDescriptions(serviceSPI, Phase.AFTER);
			if (!afterHandlers.isEmpty()) {
				handlers.put(After.class.getSimpleName(), afterHandlers);
			}

			return handlers;
		}

		return null;
	}

	static class HandlerDescription {
		public String handler;
		public String predicate;
	}

	private static List<HandlerDescription> createHandlerDescriptions(ServiceSPI serviceSPI, Phase phase) {
		List<HandlerDescription> handlers = new ArrayList<>();

		serviceSPI.registrations(phase).forEach(r -> {
			HandlerDescription description = new HandlerDescription();

			Handler handler = r.getHandler();
			description.handler = handler.toString();

			EventPredicate predicate = r.getEventPredicate();
			if (predicate == EventPredicate.ALL) {
				description.predicate = "event '*' on entity '*'";
			} else {
				description.predicate = r.getEventPredicate().toString();
			}

			handlers.add(description); // note: the handlers are already ordered within the sequence
		});

		return handlers;
	}
}
