/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.auditlog.Access;
import com.sap.cds.services.auditlog.Attribute;
import com.sap.cds.services.runtime.CdsRuntime;

class PersonalDataAnalyzerAccess extends PersonalDataAnalyzer {
	private final Map<String, Pair<CdsStructuredType, List<CqnElementRef>>> accessedSensitiveElements;

	PersonalDataAnalyzerAccess(CdsEntity entity, Map<String, Pair<CdsStructuredType, List<CqnElementRef>>> accessedSensitiveElements,
			PersonalDataCaches caches, CdsRuntime runtime) {
		super(entity, caches, runtime);
		this.accessedSensitiveElements = Objects.requireNonNull(accessedSensitiveElements, "accessedSensitiveElements must not be null");
	}

	/**
	 * Indicates whether the given {@link CdsEntity entity} contains personal data or not.
	 *
	 * @return <code>true</code> if given {@link CdsEntity entity} contains personal data.
	 */
	@Override
	boolean hasPersonalData() {
		return !this.accessedSensitiveElements.isEmpty();
	}

	/**
	 * @return a {@link List} with accesses to personal data fields.
	 */
	List<Access> getDataAccesses() {
		List<Access> accesses = new ArrayList<>();

		DataProcessor.create().action(new DataProcessor.Action() {
			@Override
			public void entries(Path path, CdsElement element, CdsStructuredType currentEntity, Iterable<Map<String, Object>> dataList) {
				PersonalDataMeta meta = getMeta(currentEntity);

				if (meta != null && personalData.getLogRead().isEnabled() && accessedSensitiveElements.containsKey(currentEntity.getQualifiedName())) {
					List<CqnElementRef> accessedList = accessedSensitiveElements.get(currentEntity.getQualifiedName()).getRight();
					List<Attribute> attributes = createAttributes(accessedList, meta);

					for (Map<String, Object> dataRow : dataList) {
						// if no attribute with sensitive data is read, there is nothing to audit
						if (!attributes.isEmpty()) {
							Access access = Access.create();
							access.setDataObject(createDataObject(dataRow, meta));
							access.setAttributes(attributes);
							access.setDataSubject(createDataSubject(dataRow, meta));
							accesses.add(access);
						}
					}
				}
			}
		}).process(super.data, super.entity);

		return accesses;
	}

	private List<Attribute> createAttributes(List<CqnElementRef> accessedElements, PersonalDataMeta meta) {
		Set<String> sensitiveDataNames = meta.getSensitiveDataNames();
		return accessedElements.stream().map(CqnReference::lastSegment).filter(sensitiveDataNames::contains).map(element -> {
			Attribute attribute = Attribute.create();
			attribute.setName(element);
			return attribute;
		}).collect(Collectors.toList());
	}

}
