/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.Map;
import java.util.Objects;

import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.NotNullValidator;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Handler class to check mandatory elements Mandatory elements are expected to
 * be sent by the client If mandatory elements are string based, empty strings
 * are considered invalid as well
 */
@ServiceName(value = "*", type = ApplicationService.class)
public class MandatoryHandler implements EventHandler {

	@Before
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		NotNullValidator.runNotNullCheck(context, false,
				MandatoryHandler::isMandatory, MandatoryHandler::isNull);
	}

	private static boolean isMandatory(CdsElement element) {
		boolean mandatory = CdsAnnotations.MANDATORY.isTrue(element);
		if (!mandatory) {
			Map<?, ?> commonFieldControl = CdsAnnotations.COMMON_FIELDCONTROL.getOrDefault(element);
			if ((commonFieldControl == null || !"Mandatory".equals(commonFieldControl.get("#")))) {
				return false;
			}
		}
		return true;
	}

	private static boolean isNull(Object value) {
		return Objects.isNull(value) || value instanceof String s && StringUtils.isEmpty(s);
	}

}
