/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import static com.sap.cds.services.utils.model.CdsAnnotations.ASSERT_RANGE;

import java.util.List;
import java.util.stream.Collectors;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.CdsDataProcessor.Validator;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.ValidatorErrorUtils;
import com.sap.cds.services.impl.utils.ValidatorExecutor;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.util.CdsTypeUtils;

@ServiceName(value = "*", type = ApplicationService.class)
public class RangeAssertionHandler implements EventHandler {

	@Before
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		CdsDataProcessor cdsDataProcessor = CdsDataProcessor.create()
			.addValidator(RangeAssertionHandler::hasAssertRange, rangeValidator(context));
		ValidatorExecutor.processEntities(cdsDataProcessor, context);
	}

	@SuppressWarnings("unchecked")
	private Validator rangeValidator(EventContext context) {
		return (path, element, value) -> {
			if (value != null) {
				Object rangeValues = ASSERT_RANGE.getListOrValue(element, null);
				List<Comparable<Object>> rangeConvertedValues = getConvertedRange(context, (List<Object>) rangeValues,
						element, element.getDeclaringType().getName());

				if (!rangeConvertedValues.isEmpty()) {
					Comparable<Object> minValue = rangeConvertedValues.get(0);
					Comparable<Object> maxValue = rangeConvertedValues.get(1);

					Comparable<Object> elementValue = (Comparable<Object>) value;

					if (elementValue.compareTo(maxValue) > 0 || elementValue.compareTo(minValue) < 0) {
						ValidatorErrorUtils.handleValidationError(context, path, element,
								CdsErrorStatuses.VALUE_OUT_OF_RANGE, element.getName(), minValue, maxValue);
					}
				}
			}
		};
	}

	private static boolean hasAssertRange(Path path, CdsElement element, CdsType type) {
		Object rangeValues = ASSERT_RANGE.getListOrValue(element, null);
		return rangeValues instanceof List l && l.size() == 2 && isAllowedType(element.getType());
	}

	@SuppressWarnings("unchecked")
	private static List<Comparable<Object>> getConvertedRange(EventContext context, List<Object> rangeValues,
			CdsElement element, String entityName) {
		try {
			return rangeValues.stream()
					.map(oValue -> (Comparable<Object>) CdsTypeUtils
							.parse(element.getType().as(CdsSimpleType.class).getType(), oValue.toString()))
					.collect(Collectors.toList());
		} catch (CdsDataException e) {
			// Throw an exception if the annotation values do not match the element
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_ANNOTATION, ASSERT_RANGE.toString(),
					element.getName(), entityName, e);
		}
	}

	@SuppressWarnings("deprecation")
	private static boolean isAllowedType(CdsType elementType){

		if (!(elementType instanceof CdsSimpleType)) return false;

		// Allowed Types for @assert.range
		switch (elementType.as(CdsSimpleType.class).getType()) {
		case INTEGER:
		case INTEGER64:
		case DECIMAL:
		case DECIMAL_FLOAT:
		case DOUBLE:
		case DATE:
		case TIME:
		case DATETIME:
		case TIMESTAMP:
			return true;
		default:
			return false;
		}

	}
}
