/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.handlerregistry.resolver;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;
import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.ActiveReadEventContext;
import com.sap.cds.services.draft.DraftCancelEventContext;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPatchEventContext;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftReadEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.impl.handlerregistry.AnnotationDescriptor;
import com.sap.cds.services.impl.handlerregistry.HandlerDescriptor;
import com.sap.cds.services.impl.utils.ReflectionUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.ListUtils;

/**
 * Handles handler methods returning instances of {@link Result} or {@link Row} or more generically <code>Iterable&lt;? extends Map&lt;String, ?&gt;&gt;</code> or <code>Map&lt;String, ?&gt;</code>
 * The resolver works only for {@link CqnService} events.
 */
public class ResultReturnResolver implements ReturnResolver {

	private final static Logger logger = LoggerFactory.getLogger(ResultReturnResolver.class);
	private final static Set<String> allowedEvents = Sets.newHashSet(CqnService.EVENT_READ
			, CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_DELETE, CqnService.EVENT_UPSERT
			, DraftService.EVENT_DRAFT_EDIT, DraftService.EVENT_DRAFT_PREPARE, DraftService.EVENT_DRAFT_SAVE
			, DraftService.EVENT_DRAFT_NEW, DraftService.EVENT_DRAFT_PATCH, DraftService.EVENT_DRAFT_CANCEL
			, DraftService.EVENT_DRAFT_CREATE, DraftService.EVENT_DRAFT_READ, DraftService.EVENT_ACTIVE_READ);
	private final static HashMap<Type, ResultReturnResolver> cache = new HashMap<>();

	/**
	 * True, if the class and generic type are applicable for this resolver
	 * @param type the generic return type
	 * @return true, if the class is equal to {@link Result} or {@link Row} or more generically <code>Iterable&lt;? extends Map&lt;String, ?&gt;&gt;</code> or <code>Map&lt;String, ?&gt;</code>
	 */
	public static ResultReturnResolver createIfApplicable(Type type) {
		ResultReturnResolver cached;
		if((cached = cache.get(type)) != null) {
			return cached;
		}

		try {
			if(isApplicableIterableType(type)) {
				return new ResultReturnResolver(true, type);
			} else if (isApplicableMapType(type)) {
				return new ResultReturnResolver(false, type);
			}
		} catch (IllegalArgumentException | ClassNotFoundException e) {
			logger.debug("Could not resolve generic type '{}'", type.getTypeName(), e);
		}

		return null;
	}

	/**
	 * True, if the type fits to <code>Iterable&lt;Map&lt;String, ?&gt;&gt;</code>
	 * @param iterableType type to be checked
	 * @return true, if the type matches <code>Iterable&lt;Map&lt;String, ?&gt;&gt;</code>
	 * @throws ClassNotFoundException
	 */
	private static boolean isApplicableIterableType(Type iterableType) throws ClassNotFoundException {
		if(iterableType == null) {
			return false;
		}

		if (ReflectionUtils.isParameterizedType(iterableType)) {
			ParameterizedType paramIterableType = ReflectionUtils.getParameterizedType(iterableType);
			// check if type is Iterable
			if(Iterable.class.isAssignableFrom(ReflectionUtils.getClassForType(paramIterableType.getRawType()))) {
				// check generic parameter of Iterable -> should be valid Map type
				if(paramIterableType.getActualTypeArguments().length == 1 && isApplicableMapType(paramIterableType.getActualTypeArguments()[0])) {
					return true;
				}
			}
		} else {
			// check if the type subclasses Iterable<...>
			Class<?> iterableExtender = ReflectionUtils.getClassForType(iterableType);
			if(Iterable.class.isAssignableFrom(iterableExtender)) {
				// walk generic interfaces and superclass and see if one returns true
				boolean applicableIterable = false;
				if(iterableExtender.getSuperclass() != Object.class) {
					applicableIterable = isApplicableIterableType(iterableExtender.getGenericSuperclass());
				}

				if(!applicableIterable) {
					for(Type genericInterface : iterableExtender.getGenericInterfaces()) {
						applicableIterable = isApplicableIterableType(genericInterface);
						if(applicableIterable) {
							break;
						}
					}
				}

				return applicableIterable;
			}
		}

		return false;
	}

	/**
	 * True, if the type fits to <code>Map&lt;String, ?&gt;</code>
	 * @param mapType type to be checked
	 * @return true, if the type matches <code>Map&lt;String, ?&gt;</code>
	 * @throws ClassNotFoundException
	 */
	private static boolean isApplicableMapType(Type mapType) throws ClassNotFoundException {
		if(mapType == null) {
			return false;
		}

		// check if the type is a map directly
		if(ReflectionUtils.isParameterizedType(mapType)) {
			ParameterizedType paramMapType = ReflectionUtils.getParameterizedType(mapType);
			if(Map.class.isAssignableFrom(ReflectionUtils.getClassForType(paramMapType.getRawType()))) {

				// check generic parameters of Map -> should be String, Object
				if(paramMapType.getActualTypeArguments().length == 2) {
					// only check string type, object type for second parameter does not need to be checked
					Type stringClazzType = paramMapType.getActualTypeArguments()[0];
					return String.class.isAssignableFrom(ReflectionUtils.getClassForType(stringClazzType));
				}
			}
		} else {
			// check if the type subclasses Map<String, Object>
			Class<?> mapExtender = ReflectionUtils.getClassForType(mapType);
			if(Map.class.isAssignableFrom(mapExtender)) {
				// walk generic interfaces and superclass and see if one returns true
				boolean applicableMap = false;
				if(mapExtender.getSuperclass() != Object.class) {
					applicableMap = isApplicableMapType(mapExtender.getGenericSuperclass());
				}

				if(!applicableMap) {
					for(Type genericInterface : mapExtender.getGenericInterfaces()) {
						applicableMap = isApplicableMapType(genericInterface);
						if(applicableMap) {
							break;
						}
					}
				}

				return applicableMap;
			}
		}
		return false;
	}

	private final boolean isIterable;
	private final Type genericType;

	private ResultReturnResolver(boolean isIterable, Type genericType) {
		this.isIterable = isIterable;
		this.genericType = genericType;
		cache.put(genericType, this);
	}

	@Override
	@SuppressWarnings("unchecked")
	public void resolve(Object returnValue, EventContext context) {
		Iterable<? extends Map<String, ?>> iterable;
		if(!isIterable) {
			iterable = ListUtils.getList((Map<String, Object>) returnValue);
		} else {
			iterable = (Iterable<Map<String, Object>>) returnValue;
		}

		switch (context.getEvent()) {
		case CqnService.EVENT_READ:
			context.as(CdsReadEventContext.class).setResult(iterable);
			break;
		case CqnService.EVENT_CREATE:
			context.as(CdsCreateEventContext.class).setResult(iterable);
			break;
		case CqnService.EVENT_UPDATE:
			context.as(CdsUpdateEventContext.class).setResult(iterable);
			break;
		case CqnService.EVENT_DELETE:
			context.as(CdsDeleteEventContext.class).setResult(iterable);
			break;
		case CqnService.EVENT_UPSERT:
			context.as(CdsUpsertEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_EDIT:
			context.as(DraftEditEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_PREPARE:
			context.as(DraftPrepareEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_SAVE:
			context.as(DraftSaveEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_NEW:
			context.as(DraftNewEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_PATCH:
			context.as(DraftPatchEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_CANCEL:
			context.as(DraftCancelEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_CREATE:
			context.as(DraftCreateEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_DRAFT_READ:
			context.as(DraftReadEventContext.class).setResult(iterable);
			break;
		case DraftService.EVENT_ACTIVE_READ:
			context.as(ActiveReadEventContext.class).setResult(iterable);
			break;
		default:
			// nothing to do
		}
	}

	@Override
	public void verifyOrThrow(HandlerDescriptor descriptor) {
		Set<String> events = new HashSet<>();
		for(AnnotationDescriptor ad : descriptor.getAnnotations()) {
			events.addAll(Arrays.asList(ad.getEvents()));
		}

		for(String event : events) {
			verifyEventOrThrow(event, descriptor);
		}
	}

	private void verifyEventOrThrow(String event, HandlerDescriptor descriptor) {
		if(!allowedEvents.contains(event)) {
			throw new ErrorStatusException(CdsErrorStatuses.RETURN_TYPE_MISMATCH, event, genericType.getTypeName(), descriptor.getMethodName());
		}
	}

}
