/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox;

import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Statement;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.Upsert;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.Service;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.outbox.OutboxMessage;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;

@ServiceName(value = "*", type = OutboxService.class)
public class CqnServiceOutboxHandler implements EventHandler {
	@Before
	private void publishedByOutbox(OutboxMessageEventContext context) {
		Service service = context.getServiceCatalog().getService(context.getEvent());
		if(Boolean.FALSE.equals(context.getIsInbound()) || !(service instanceof CqnService)) {
			return;
		}

		OutboxMessage message = context.getMessage();
		if (message.getParams().get("cqn") instanceof String cqnQuery) {
			// Parse CQN Query
			Statement<?> cqn = parseCqn(message.getEvent(), cqnQuery, context.getModel());
			message.getParams().put("cqn", cqn);
		}
	}

	@Before
	private void adjustOutboxMessage(OutboxMessageEventContext context) {
		// prepare outbox message for storage only
		Service service = context.getServiceCatalog().getService(context.getEvent());
		if(Boolean.TRUE.equals(context.getIsInbound()) || !(service instanceof CqnService)) {
			return;
		}

		OutboxMessage message = context.getMessage();
		if (message.getParams().get("cqn") instanceof CqnStatement cqn) {
			message.getParams().put("cqn", cqn.toJson());
		}
	}

	private Statement<?> parseCqn(String event, String cqnQuery, CdsModel cdsModel) {
		return switch (event) {
			case CqnService.EVENT_CREATE -> Insert.cqn(cdsModel, cqnQuery);
			case CqnService.EVENT_DELETE -> Delete.cqn(cqnQuery);
			case CqnService.EVENT_READ -> Select.cqn(cqnQuery);
			case CqnService.EVENT_UPDATE -> Update.cqn(cdsModel, cqnQuery);
			case CqnService.EVENT_UPSERT -> Upsert.cqn(cdsModel, cqnQuery);
			// could be an action
			default -> Select.cqn(cqnQuery);
		};
	}

}
