/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox.persistence.collectors;

import static com.sap.cds.services.impl.model.DynamicModelProvider.STATIC_MODEL_ACCESS_FEATURE;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.CdsLockTimeoutException;
import com.sap.cds.Result;
import com.sap.cds.Struct;
import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.Update;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.services.environment.CdsProperties.Outbox.OutboxConfig;
import com.sap.cds.services.impl.outbox.Messages;
import com.sap.cds.services.impl.outbox.Messages_;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutbox;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.outbox.OutboxMessage;
import com.sap.cds.services.outbox.OutboxMessageEventContext;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.OpenTelemetryUtils;

import io.opentelemetry.api.trace.Span;
import io.opentelemetry.api.trace.SpanKind;
import io.opentelemetry.context.Scope;


/**
 * Outbox collector implementation for a specific outbox partition.
 */
public class PartitionCollector implements Runnable {

	private static final Logger LOG = LoggerFactory.getLogger(PartitionCollector.class);

	private PersistenceService db;
	private final CdsRuntime runtime;
	private final OutboxService outboxService;
	private final int chunkSize;
	private final String target;

	private final Object pauseMonitor = new Object();
	private final AtomicInteger pauseCount = new AtomicInteger(5);
	private volatile boolean pause = false; // flag to synchronize wakeup

	private final long maxPauseMillis;
	private final long emitTimeoutSeconds;
	private final int maxPublishAttempts;
	private final boolean storeLastError;
	private final boolean ordered;

	private final Supplier<List<TenantInfo>> tenantSupplier;

	public PartitionCollector(CdsRuntime runtime, PersistentOutbox outboxService, OutboxConfig config, boolean ordered, Supplier<List<TenantInfo>> tenantSupplier) {
		this.runtime = runtime;
		this.outboxService = outboxService;
		this.target = outboxService.getName();

		this.chunkSize = config.getChunkSize();
		this.maxPauseMillis = config.getMaxPause().getSeconds() * 1000;
		this.emitTimeoutSeconds = config.getEmitTimeout().getSeconds();
		this.maxPublishAttempts = config.getMaxAttempts();
		this.storeLastError = config.getStoreLastError().isEnabled();
		this.ordered = ordered;

		this.tenantSupplier = tenantSupplier;

		if (!storeLastError) {
			LOG.debug("Storing errors for outbox '{}' is disabled.", outboxService.getName());
		}
	}

	@Override
	public void run() {
		this.db = runtime.getServiceCatalog().getService(PersistenceService.class, PersistenceService.DEFAULT_NAME);
		processPartition();
	}

	private void pause() {
		synchronized(pauseMonitor) {
			pause = true;
			try {
				long pauseInMillis = getPauseMillis(pauseCount.get(), maxPauseMillis);
				LOG.debug("Pausing collector '{}' for {} ms", target, pauseInMillis);
				pauseMonitor.wait(pauseInMillis);
			} catch (InterruptedException e) {
				LOG.debug("Collector '{}' interrupted", target);
				Thread.currentThread().interrupt();
			}
			pause = false;
		}
	}

	public void unpause() {
		// ensures that the next pause is short
		pauseCount.set(0);
		// wakes up a currently sleeping collector
		if (pause) {
			synchronized (pauseMonitor) {
				if(pause) {
					pause = false;
					pauseMonitor.notifyAll();
					LOG.debug("Notified paused collector '{}'", target);
				}
			}
		}
	}


	private boolean isNotEmptyOutbox(String tenant) {
		LOG.debug("Checking tenant '{}' for outbox entries in collector '{}'", tenant, target);
		return runtime.requestContext().featureToggles(STATIC_MODEL_ACCESS_FEATURE).systemUser(tenant).run(req -> {
			CqnSelect select = Select.from(Messages_.class)
					.where(e -> e.target().eq(target)
							.and(e.attempts().lt(this.maxPublishAttempts)))
					.limit(1);
			Result res = db.run(select);
			return res.rowCount() != 0;
		});
	}

	private void processPartition() {
		while (!Thread.currentThread().isInterrupted()) {
			try {
				LOG.debug("Executing collector '{}'", target);
				AtomicBoolean doPause = new AtomicBoolean(true);
				List<TenantInfo> tenantInfos = new ArrayList<>(this.tenantSupplier.get());
				// shuffle the list to avoid processing the same tenant first every time
				Collections.shuffle(tenantInfos);
				for (TenantInfo tenantInfo : tenantInfos) {
					Optional<Span> span = OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.OUTBOX, SpanKind.SERVER);
					try (Scope scope = span.map(Span::makeCurrent).orElse(null)) {
						span.ifPresent(s -> {
							s.updateName("Outbox Collector (" + target + ")");
							s.setAttribute(OpenTelemetryUtils.CDS_TENANT, tenantInfo.getTenant());
							s.setAttribute(OpenTelemetryUtils.CDS_OUTBOX_TARGET, target);
						});
						if (isNotEmptyOutbox(tenantInfo.getTenant())) {
							LOG.debug("Processing tenant '{}' in collector '{}'", tenantInfo.getTenant(), target);
							boolean interrupted = runtime.requestContext().systemUser(tenantInfo.getTenant()).run(req -> {
								return runtime.changeSetContext().run(ctx -> {
									Predicate where = CQL.get(Messages.TARGET).eq(target);
									// for compatibility
									if (OutboxService.PERSISTENT_UNORDERED_NAME.equals(target)) {
										where = CQL.or(where, CQL.get(Messages.TARGET).startsWith("auditlog/"));
									} else if (OutboxService.PERSISTENT_ORDERED_NAME.equals(target)) {
										where = CQL.or(where, CQL.get(Messages.TARGET).startsWith("messaging/"));
									}
									where = CQL.and(where, CQL.get(Messages.ATTEMPTS).lt(this.maxPublishAttempts));

									long skip = calculateOffset(where);
									CqnSelect select = Select.from(Messages_.class).where(where)
											.orderBy(e -> e.timestamp().asc(), e -> e.ID().asc())
											.limit(this.chunkSize, skip)
											.lock(0);

									Result res = db.run(select);
									// at least one tenant still has more messages to process
									if(skip != 0 || res.rowCount() >= select.top()) {
										doPause.set(false);
									}

									if(res.rowCount() > 0) {
										// track start of dispatch to interrupt if sequential dispatching takes too long
										final Instant startOfDispatch = Instant.now();
										for (Messages msg : res.listOf(Messages.class)) {
											// sequential publishing
											PublishState state = publish(msg, startOfDispatch);

											if (state == PublishState.SUCCESS) {
												db.run(Delete.from(Messages_.class).where(e -> e.ID().eq(msg.getId())));
											} else if (state == PublishState.TIMEOUT) {
												break;
											} else if (state == PublishState.INTERRUPTED) {
												return true;
											}

											// track time of dispatch process and interrupt if threshold has been reached
											if (Duration.between(startOfDispatch, Instant.now()).getSeconds() > this.emitTimeoutSeconds) {
												break;
											}
										}
									}
									return false;
								});
							});
							if (interrupted) {
								doPause.set(false);
								break;
							}
						} else {
							LOG.debug("The outbox for the tenant '{}' in collector '{}' is empty", tenantInfo.getTenant(), target);
						}
					} catch (Exception e) {
						OpenTelemetryUtils.recordException(span, e);

						if (isLockTimeoutException(e)) {
							LOG.debug("Collector '{}' timed out waiting for table lock for tenant '{}'", target, tenantInfo.getTenant());
							continue;
						}
						LOG.warn("Exception occurred for tenant '{}' in collector '{}'", tenantInfo.getTenant(), target, e);
					} finally {
						OpenTelemetryUtils.endSpan(span);
					}
				}

				if(doPause.get()) {
					pause();
					if(pauseCount.get() < 20) {
						pauseCount.addAndGet(2);
					}
				} else {
					pauseCount.set(0);
				}
			} catch (Throwable e) {
				LOG.warn("Unexpected exception occured in collector '{}'", target, e);
			}
		}
	}

	private long calculateOffset(Predicate whereClause) {
		if (this.ordered) {
			return 0;
		}

		CqnSelect select = Select
				.from(Messages_.class)
				.columns(c -> CQL.count().as("count"))
				.where(whereClause);
		Result res = db.run(select);
		long count = ((Number) res.single().get("count")).longValue();
		long chunks = count / this.chunkSize;
		// no secure random number needed, because the offset is not used for security relevant operations
		long offset = chunks < 2 ? 0 : ThreadLocalRandom.current().nextLong(chunks) * this.chunkSize; // NOSONAR

		LOG.debug("Calculated offset for unordered processing of outbox collector '{}' is {}", target, offset);

		return offset;
	}

	@SuppressWarnings("unchecked")
	private PublishState publish(final Messages msg, final Instant startOfDispatch) {
		// by publishing a retry message we have to check whether the retry pause is reached before trying
		if (msg.getAttempts() != 0 && msg.getLastAttemptTimestamp() != null && (Duration.between(msg.getLastAttemptTimestamp(), Instant.now()).toMillis() < getPauseMillis(msg.getAttempts(), Integer.MAX_VALUE))) {
			LOG.debug("Message '{}' cannot be republished until the retry waiting time is reached", msg.getId());
			return PublishState.TIMEOUT;
		}

		String outboxEvent = null;
		Map<String, Object> message = JsonParser.map(JsonParser.parseJson(msg.getMsg()));
		// backward compatibility
		if (msg.getTarget().startsWith("messaging/") ||
				msg.getTarget().startsWith("auditlog/")) {
			outboxEvent = msg.getTarget().substring(msg.getTarget().indexOf('/') + 1);
		} else {
			outboxEvent = (String) message.get(PersistentOutbox.ATTR_EVENT);
			message = (Map<String, Object>) message.get(PersistentOutbox.ATTR_MESSAGE);
		}

		LOG.debug("Publishing outbox message for target '{}' with event '{}'", msg.getTarget(), outboxEvent);

		OutboxMessageEventContext ctx = OutboxMessageEventContext.create(outboxEvent);
		ctx.setIsInbound(true);
		ctx.setTimestamp(msg.getTimestamp());
		ctx.setMessage(Struct.access(message).as(OutboxMessage.class));

		// recover the boxed context
		while (true) {
			try {
				outboxService.emit(ctx);
				return PublishState.SUCCESS;
			} catch (Throwable e) { // NOSONAR
				// we should may be check the CdsErrorStatuses.NO_ON_HANDLER exception in order
				// to ignore the entry without handling it as an error.
				LOG.warn("Failed to emit Outbox message with id '{}' for target '{}' with event '{}'", msg.getId(), msg.getTarget(), outboxEvent, e);

				int currentAttempts = msg.getAttempts();

				// re-attempt to publish
				msg.setAttempts(++currentAttempts);
				msg.setLastAttemptTimestamp(Instant.now());
				Map<String, Object> data = new HashMap<>();
				data.put(Messages.ATTEMPTS, msg.getAttempts());
				data.put(Messages.LAST_ATTEMPT_TIMESTAMP, msg.getLastAttemptTimestamp());

				if (storeLastError) {
					StringWriter stringWriter = new StringWriter();
					e.printStackTrace(new PrintWriter(stringWriter));

					data.put(Messages.LAST_ERROR, stringWriter.toString());
				}

				db.run(Update.entity(Messages_.class).data(data).where(m -> m.ID().eq(msg.getId())));

				if (currentAttempts < this.maxPublishAttempts) {
					try {
						// exponential backoff in ms for re-attempt
						long pauseInMillis = getPauseMillis(currentAttempts, Integer.MAX_VALUE);

						// check time + pause of dispatch process and interrupt if threshold has been reached
						if (Duration.between(startOfDispatch, Instant.now().plusMillis(pauseInMillis)).getSeconds() > this.emitTimeoutSeconds) {
							LOG.debug("The retry waiting time of message '{}' would exceed the emit timeout, therefore release lock and commit transaction", msg.getId());
							return PublishState.TIMEOUT;
						}
						// wait till next try
						TimeUnit.MILLISECONDS.sleep(pauseInMillis);
					} catch (InterruptedException ie) {
						Thread.currentThread().interrupt();
						return PublishState.INTERRUPTED;
					}
				} else {
					// giving up to publish
					LOG.warn("Reached maximum number of attempts to emit Outbox message with id '{}' to target '{}' with event '{}'",
							msg.getId(), msg.getTarget(), outboxEvent);
					return PublishState.FAILED;
				}
			}
		}
	}

	private static long getPauseMillis(int pauseCount, long maxTimeoutMillis) {
		long retryInMillis = Math.round(Math.pow(2d, pauseCount) * 1000 + ThreadLocalRandom.current().nextLong(1001));
		return Math.min(retryInMillis, maxTimeoutMillis);
	}

	private static boolean isLockTimeoutException(Throwable t) {
		while (t != null) {
			if (t instanceof CdsLockTimeoutException) {
				return true;
			}
			t = t.getCause();
		}
		return false;
	}

	private static enum PublishState {
		SUCCESS,
		FAILED,
		TIMEOUT,
		INTERRUPTED
	}
}
