/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import static com.sap.cds.reflect.CdsBaseType.UUID;
import static java.util.UUID.randomUUID;

import java.util.List;
import java.util.Map;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = PersistenceService.class)
public class GenerateUUIDHandler implements EventHandler {

	private static final CdsDataProcessor uuidKeyGenerator = CdsDataProcessor.create().addGenerator(
			(p, element, type) -> element.isKey() && type.isSimpleType(UUID), //
			(p, element, isNull) -> randomUUID().toString());

	/**
	 * Make sure, that any entity that is going to be inserted in the given create
	 * event context has a value set in the given key element of type "cds.UUID". If
	 * no value is currently set, a new UUID is generated.
	 *
	 * @param context the create event context
	 */
	@Before(event = { CqnService.EVENT_CREATE })
	@HandlerOrder(OrderConstants.Before.CALCULATE_FIELDS)
	public void fillUUID(EventContext context) {
		// if the entity that is going to be inserted has a key element of type
		// "cds.UUID", generate any missing UUIDs
		List<Map<String, Object>> entries = CdsServiceUtils.getEntities(context);
		uuidKeyGenerator.process(entries, context.getTarget());
	}

}
