/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.request;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.services.request.FeatureToggle;
import com.sap.cds.services.request.FeatureTogglesInfo;

public class FeatureTogglesInfoFactory {

	public static FeatureTogglesInfo create(List<FeatureToggle> featureToggles) {
		return new FeatureTogglesInfoImpl(featureToggles.stream().collect(Collectors.toMap(ft -> ft.getName(), ft -> ft)));
	}

	public static FeatureTogglesInfo create(Map<String, Boolean> featureToggles) {
		return new FeatureTogglesInfoImpl(featureToggles.entrySet().stream()
				.collect(Collectors.toMap(e -> e.getKey(), e -> new FeatureToggleImpl(e.getKey(), e.getValue()))));
	}

	private static class FeatureTogglesInfoImpl implements FeatureTogglesInfo {

		private final Map<String, FeatureToggle> features;

		private FeatureTogglesInfoImpl(Map<String, FeatureToggle> features) {
			this.features = features;
		}

		@Override
		public boolean isEnabled(String feature) {
			return features.containsKey(feature) && features.get(feature).isEnabled();
		}

		@Override
		public Stream<FeatureToggle> getFeatureToggles() {
			return features.values().stream();
		}

		@Override
		public Stream<FeatureToggle> getEnabledFeatureToggles() {
			return getFeatureToggles().filter(FeatureToggle::isEnabled);
		}

		@Override
		public FeatureToggle getFeatureToggle(String feature) {
			return features.get(feature);
		}

	}

	private static class FeatureToggleImpl implements FeatureToggle {

		private final String name;
		private final boolean enabled;

		public FeatureToggleImpl(String name, boolean enabled) {
			this.name = name;
			this.enabled = enabled;
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public boolean isEnabled() {
			return enabled;
		}
	}
}
