package com.sap.cds.services.impl.utils;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.CdsDataProcessor.Filter;
import com.sap.cds.CdsDataProcessor.Mode;
import com.sap.cds.CdsDataProcessor.Validator;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.impl.parser.PathParser;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.model.CdsAnnotations;

public class NotNullValidator {

	private NotNullValidator() {
		// empty
	}

	public static void runNotNullCheck(EventContext context, boolean forceException) {
		runNotNullCheck(context, forceException, e -> e.isNotNull() && CdsAnnotations.ASSERT_NOTNULL.isTrue(e), Objects::isNull);
	}

	public static void runNotNullCheck(EventContext context, boolean forceException, Predicate<CdsElement> requiredByModel, Predicate<Object> isNull) {
		DataProcessor processor = DataProcessor.create()
				.addValidator(requiresNotNullCheck(requiredByModel), assertNotNull(context, forceException, isNull), Mode.DECLARED);
		if (context.getEvent().equals(CqnService.EVENT_CREATE)) {
			processor.forInsert();
		} else if (context.getEvent().equals(CqnService.EVENT_UPDATE) || context.getEvent().equals(CqnService.EVENT_UPSERT)) {
			processor.forUpdate();
		}
		ValidatorExecutor.processResolvedEntities(processor, context);
	}

	private static Filter requiresNotNullCheck(Predicate<CdsElement> requiredByModel) {
		return (path, element, type) -> {
			// elements with default values don't have to be checked
			if (element == null || element.getType().isSimple() && element.defaultValue().isPresent()) {
				return false;
			}

			// generated fk elements are handled separately
			if (CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrValue(element, null) != null) {
				return false;
			}

			if (!requiredByModel.test(element)) {
				return false;
			}

			Iterator<ResolvedSegment> segments = path.reverse();
			String lastSegment = segments.next().segment().id();
			if (segments.hasNext()) {
				Optional<CdsElement> optAssociation = segments.next().type().findAssociation(lastSegment);
				if (optAssociation.isPresent()) {
					CdsElement association = optAssociation.get();
					List<CdsElement> keyElements = CdsModelUtils.getAssociationKeys(association);
					if (keyElements.contains(element) || keyElements.stream().anyMatch(k -> element.getName().equals(CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrDefault(k)))) {
						// key elements of the association target are not mandatory,
						// as they get auto-filled
						return false;
					}
				}
			}
			return true;
		};
	}

	/**
	 * Returns a validator for mandatory annotated elements or elements with not
	 * null constraints. Validated elements can be of type association and requires
	 * a not null (or empty string) check on its foreign key elements.
	 *
	 * @param context      the event context
	 * @param isNull       predicate checking whether an object is set to null (or
	 *                     an empty string)
	 * @param errorHandler error handler that takes the element and
	 *                     entity name that is validated
	 * @return a validator implementation
	 */
	private static Validator assertNotNull(EventContext context, boolean forceException, Predicate<Object> isNull) {
		return (path, element, value) -> {

			// To check / possible scenarios:
			// - element explicitly set to null (or empty string)
			// - insert with missing non-association element
			// - insert with missing association element
			// - (partial) update with missing association element

			boolean isPartialUpdate = CqnService.EVENT_UPDATE.equals(context.getEvent());
			boolean isAssociation = element.getType().isAssociation();
			boolean notInRequestBody = value == CdsDataProcessor.ABSENT;

			// mandatory element explicitly set to null (or empty string), or (-> error)
			// insert with a missing non-association element                  (-> error)
			if (isNull.test(value) || !isPartialUpdate && !isAssociation && notInRequestBody) {
				handleNotNullError(context, forceException, path, element, element.getDeclaringType());

			// insert with missing association element, or       (-> check fk elements)
			// (partial) update with missing association element (-> check fk elements)
			} else if (notInRequestBody) {
				for (CdsElement e : invalidFkElementsOfFwdAssociation(context.getModel(), isPartialUpdate, path, element, isNull)) {
					handleNotNullError(context, forceException, path, e, element.getDeclaringType());
				}
			}
			// (partial) update with missing non-association element (-> ignored)
		};
	}

	// returns all invalid fk elements of a forward association element, filtering
	// them with the given "isNull" predicate
	//
	// TODO: revisit behavior of OnConditionAnalyzer for structured OData
	private static List<CdsElement> invalidFkElementsOfFwdAssociation(CdsModel model, boolean isPartialUpdate, 
			Path path, CdsElement element, Predicate<Object> isNull) {

		boolean isFwdAssociation = element.getType().isAssociation()
				&& !com.sap.cds.util.CdsModelUtils.isReverseAssociation(element);

		// make sure to validate only forward associations
		if (isFwdAssociation) {
			Map<String, Object> parentValues = path.target().values();

			Set<String> fkElements = CdsModelUtils.getFkMapping(model, element, false).keySet();

			return fkElements.stream().filter(e -> {
				if (!isPartialUpdate) {
					// insert with missing fk element, or explicitly set to null or empty string
					return !parentValues.containsKey(e) || isNull.test(parentValues.get(e));
				} else {
					// (partial) update with fk element explicitly set to null or empty string
					return parentValues.containsKey(e) && isNull.test(parentValues.get(e));
				}
			}).map(e -> com.sap.cds.util.CdsModelUtils.element(element.getDeclaringType(), PathParser.segments(e)))
					.collect(Collectors.toList());
		}
		return Collections.emptyList();
	}

	private static void handleNotNullError(EventContext context, boolean forceException, Path path, CdsElement element, CdsStructuredType entity) {
		if (CdsModelUtils.isInternalOperationType(element.getDeclaringType().as(CdsDefinition.class))) {
			ValidatorErrorUtils.handleValidationError(context, forceException, path, element, CdsErrorStatuses.MISSING_VALUE, element.getName());
		} else {
			ValidatorErrorUtils.handleValidationError(context, forceException, path, element, CdsErrorStatuses.VALUE_REQUIRED, element.getName(), entity.getQualifiedName());
		}
	}

}
