/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl;

import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import com.sap.cds.Struct;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.EventName;
import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.auditlog.ConfigChangeLogContext;
import com.sap.cds.services.auditlog.DataAccessLogContext;
import com.sap.cds.services.auditlog.DataModificationLogContext;
import com.sap.cds.services.auditlog.SecurityLogContext;
import com.sap.cds.services.authentication.AuthenticationInfo;
import com.sap.cds.services.authorization.CalcWhereConditionEventContext;
import com.sap.cds.services.authorization.EntityAccessEventContext;
import com.sap.cds.services.authorization.FunctionAccessEventContext;
import com.sap.cds.services.authorization.GetRestrictionEventContext;
import com.sap.cds.services.authorization.ServiceAccessEventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.draft.ActiveReadEventContext;
import com.sap.cds.services.draft.DraftCancelEventContext;
import com.sap.cds.services.draft.DraftCreateEventContext;
import com.sap.cds.services.draft.DraftEditEventContext;
import com.sap.cds.services.draft.DraftGcEventContext;
import com.sap.cds.services.draft.DraftNewEventContext;
import com.sap.cds.services.draft.DraftPatchEventContext;
import com.sap.cds.services.draft.DraftPrepareEventContext;
import com.sap.cds.services.draft.DraftReadEventContext;
import com.sap.cds.services.draft.DraftSaveEventContext;
import com.sap.cds.services.impl.auditlog.ConfigChangeLogContextImpl;
import com.sap.cds.services.impl.auditlog.DataAccessLogContextImpl;
import com.sap.cds.services.impl.auditlog.DataModificationLogContextImpl;
import com.sap.cds.services.impl.auditlog.SecurityLogContextImpl;
import com.sap.cds.services.impl.authorization.CalcWhereConditionEventContextImpl;
import com.sap.cds.services.impl.authorization.EntityAccessEventContextImpl;
import com.sap.cds.services.impl.authorization.FunctionAccessEventContextImpl;
import com.sap.cds.services.impl.authorization.GetRestrictionEventContextImpl;
import com.sap.cds.services.impl.authorization.ServiceAccessEventContextImpl;
import com.sap.cds.services.impl.cds.CdsCreateEventContextImpl;
import com.sap.cds.services.impl.cds.CdsDeleteEventContextImpl;
import com.sap.cds.services.impl.cds.CdsReadEventContextImpl;
import com.sap.cds.services.impl.cds.CdsUpdateEventContextImpl;
import com.sap.cds.services.impl.cds.CdsUpsertEventContextImpl;
import com.sap.cds.services.impl.draft.ActiveReadEventContextImpl;
import com.sap.cds.services.impl.draft.DraftCancelEventContextImpl;
import com.sap.cds.services.impl.draft.DraftCreateEventContextImpl;
import com.sap.cds.services.impl.draft.DraftEditEventContextImpl;
import com.sap.cds.services.impl.draft.DraftGcEventContextImpl;
import com.sap.cds.services.impl.draft.DraftNewEventContextImpl;
import com.sap.cds.services.impl.draft.DraftPatchEventContextImpl;
import com.sap.cds.services.impl.draft.DraftPrepareEventContextImpl;
import com.sap.cds.services.impl.draft.DraftReadEventContextImpl;
import com.sap.cds.services.impl.draft.DraftSaveEventContextImpl;
import com.sap.cds.services.impl.request.RequestContextImpl;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.impl.utils.NameOnlyCdsEntity;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ClassMethods;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.ResultUtils;
import com.sap.cds.services.utils.StringUtils;


public class EventContextImpl implements EventContextSPI {

	private final static ClassMethods eventContextImplMethods = ClassMethods.create(EventContextImpl.class);

	private final static Map<Class<? extends EventContext>, Function<EventContext, ? extends EventContext>> implClasses = new HashMap<>();
	{
		//cds
		implClasses.put(CdsCreateEventContext.class, CdsCreateEventContextImpl::new);
		implClasses.put(CdsDeleteEventContext.class, CdsDeleteEventContextImpl::new);
		implClasses.put(CdsReadEventContext.class, CdsReadEventContextImpl::new);
		implClasses.put(CdsUpdateEventContext.class, CdsUpdateEventContextImpl::new);
		implClasses.put(CdsUpsertEventContext.class, CdsUpsertEventContextImpl::new);

		// authorization
		implClasses.put(CalcWhereConditionEventContext.class, CalcWhereConditionEventContextImpl::new);
		implClasses.put(EntityAccessEventContext.class, EntityAccessEventContextImpl::new);
		implClasses.put(FunctionAccessEventContext.class, FunctionAccessEventContextImpl::new);
		implClasses.put(GetRestrictionEventContext.class, GetRestrictionEventContextImpl::new);
		implClasses.put(ServiceAccessEventContext.class, ServiceAccessEventContextImpl::new);

		// audit log
		implClasses.put(ConfigChangeLogContext.class, ConfigChangeLogContextImpl::new);
		implClasses.put(DataAccessLogContext.class, DataAccessLogContextImpl::new);
		implClasses.put(DataModificationLogContext.class, DataModificationLogContextImpl::new);
		implClasses.put(SecurityLogContext.class, SecurityLogContextImpl::new);

		// draft
		implClasses.put(DraftReadEventContext.class, DraftReadEventContextImpl::new);
		implClasses.put(ActiveReadEventContext.class, ActiveReadEventContextImpl::new);
		implClasses.put(DraftCancelEventContext.class, DraftCancelEventContextImpl::new);
		implClasses.put(DraftCreateEventContext.class, DraftCreateEventContextImpl::new);
		implClasses.put(DraftEditEventContext.class, DraftEditEventContextImpl::new);
		implClasses.put(DraftGcEventContext.class, DraftGcEventContextImpl::new);
		implClasses.put(DraftNewEventContext.class, DraftNewEventContextImpl::new);
		implClasses.put(DraftPatchEventContext.class, DraftPatchEventContextImpl::new);
		implClasses.put(DraftPrepareEventContext.class, DraftPrepareEventContextImpl::new);
		implClasses.put(DraftSaveEventContext.class, DraftSaveEventContextImpl::new);
	}

	private final Map<String, Object> map = new HashMap<>();

	private Service service;
	private final String event;
	private final String entityName;
	private CdsEntity target;

	private boolean completed = false;

	public EventContextImpl(String event, String entityName) {
		this.event = event;
		this.entityName = StringUtils.isEmpty(entityName) ? null : entityName;
	}

	@Override
	public CdsModel getModel() {
		return getRequestContext().getModel();
	}

	@Override
	public ServiceCatalog getServiceCatalog() {
		return getRequestContext().getServiceCatalog();
	}

	@Override
	public ParameterInfo getParameterInfo() {
		return getRequestContext().getParameterInfo();
	}

	@Override
	public UserInfo getUserInfo() {
		return getRequestContext().getUserInfo();
	}

	@Override
	public AuthenticationInfo getAuthenticationInfo() {
		return getRequestContext().getAuthenticationInfo();
	}

	@Override
	public FeatureTogglesInfo getFeatureTogglesInfo() {
		return getRequestContext().getFeatureTogglesInfo();
	}

	@Override
	public Messages getMessages() {
		return getRequestContext().getMessages();
	}

	@Override
	public CdsRuntime getCdsRuntime() {
		ServiceSPI serviceSPI = CdsServiceUtils.getServiceSPI(service);
		return serviceSPI != null ? serviceSPI.getCdsRuntime() : null;
	}

	// This throws an NPE, as long as the EventContext was not emitted on a service,
	// or more precise as long as setService() was not called
	private RequestContext getRequestContext() {
		return RequestContext.getCurrent(getCdsRuntime());
	}

	@Override
	public ChangeSetContext getChangeSetContext() {
		return ChangeSetContext.getCurrent();
	}

	@Override
	public Service getService() {
		return service;
	}

	@Override
	public void setService(Service service) {
		this.service = service;
	}

	@Override
	public String getEvent() {
		return event;
	}

	@Override
	public CdsEntity getTarget() {
		if(entityName == null) {
			return null;
		}

		if(target == null) {
			RequestContext context = RequestContextImpl.getCurrentOrNull(getCdsRuntime());
			if(context != null && context.getModel() != null) {
				target = CdsModelUtils.getEntityOrThrow(context.getModel(), entityName);
			}
		}
		return target != null ? target : new NameOnlyCdsEntity(entityName);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T extends EventContext> T as(Class<T> clazz) {
		// if this object is already of type T, simply return it
		if(clazz.isAssignableFrom(this.getClass())) {
			return (T) this;
		}

		// make sure the EventContext to be casted matches the event type
		EventName expectedEventName = clazz.getAnnotation(EventName.class);
		if (expectedEventName == null || StringUtils.isEmpty(expectedEventName.value())) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_MISSING_ANNOTATION, clazz.getName());

		} else if ( !expectedEventName.value().equals("*") && !event.equals(expectedEventName.value())) {
			throw new ErrorStatusException(CdsErrorStatuses.EVENT_CONTEXT_EVENT_MISMATCH,
					clazz.getName(), expectedEventName.value(), event);
		}

		// if we have a implementation class for type T, instantiate it
		if(implClasses.containsKey(clazz)) {
			return (T)implClasses.get(clazz).apply(this);
		}

		// interface laid over data map
		T mapAccessor = Struct.access(map).as(clazz);

		// proxy combining access to EventContext methods and methods of the typed map accessor
		return (T) Proxy.newProxyInstance(clazz.getClassLoader(), new Class[] { clazz, EventContextSPI.class },
				(proxy, method, methodArgs) -> {
					// if the method exists on the generic event context, invoke directly on this object
					Method originalMethod = eventContextImplMethods.lookupMethod(method);
					if(originalMethod != null) {
						// prevent creating further proxies
						if(method.getName().equals("as") && methodArgs.length == 1 && ((Class<?>) methodArgs[0]).isAssignableFrom(clazz)) {
							return proxy;
						}

						return originalMethod.invoke(this, methodArgs);
					}

					// TODO generalize this concept
					if(method.getName().equals("setResult")) {
						Type[] types = method.getGenericParameterTypes();
						if(types.length == 1 && types[0].getTypeName().equals("java.lang.Iterable<? extends java.util.Map<java.lang.String, ?>>")) {
							methodArgs[0] = ResultUtils.convert((Iterable<? extends Map<String, ?>>) methodArgs[0]);
						}
					}

					// method only exists on the interface
					Object obj = method.invoke(mapAccessor, methodArgs);

					// TODO generalize this concept
					// automatic completion, when result is set
					if(method.getName().equals("setResult")) {
						setCompleted();
					}

					return obj;
				});
	}

	@Override
	public Set<String> keySet() {
		return Collections.unmodifiableSet(map.keySet());
	}

	@Override
	public Object get(String key) {
		return map.get(key);
	}

	@Override
	public void put(String key, Object value) {
		map.put(key, value);
	}

	@Override
	public void setCompleted() {
		this.completed = true;
	}

	@Override
	public boolean isCompleted() {
		return completed == true;
	}

}
