/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.EventContext;
import com.sap.cds.services.Service;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.handler.Handler;
import com.sap.cds.services.impl.handlerregistry.EventPredicateTools;
import com.sap.cds.services.impl.handlerregistry.HandlerRegistryTools;
import com.sap.cds.services.impl.runtime.ChangeSetContextRunnerImpl;
import com.sap.cds.services.impl.runtime.RequestContextRunnerImpl;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.impl.utils.OpenTelemetryUtils;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;

import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Scope;

public class ServiceImpl implements ServiceSPI {
	private final static Logger logger = LoggerFactory.getLogger(ServiceImpl.class);
	private final static Handler DEFAULT_REJECT = (context) -> {
		throw new ErrorStatusException(CdsErrorStatuses.NO_ON_HANDLER);
	};

	private final String name;
	private final List<Registration> beforeRegistrations = new CopyOnWriteArrayList<>();
	private final List<Registration> onRegistrations = new CopyOnWriteArrayList<>();
	private final List<Registration> afterRegistrations = new CopyOnWriteArrayList<>();
	private Service delegator;

	private CdsRuntime runtime = null;

	public ServiceImpl(String name) {
		this(name, null);
	}

	public ServiceImpl(String name, Service delegator) {
		if (StringUtils.isEmpty(name)) {
			throw new ErrorStatusException(CdsErrorStatuses.SERVICE_NAME_REQUIRED);
		}
		this.name = name.trim();
		this.delegator = delegator == null ? this : delegator;

		// register this handler now to ensure that no other handler
		// can be the first handler with order Integer.MAX_VALUE
		registerHandler(Phase.ON, EventPredicate.ALL, DEFAULT_REJECT, OrderConstants.On.REJECT);

		// register event handlers defined in this service or the delegator
		HandlerRegistryTools.registerInstance(this.delegator, this);
	}

	@Override
	public void setCdsRuntime(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public CdsRuntime getCdsRuntime() {
		return this.runtime;
	}

	@Override
	public void setDelegator(Service delegator) {
		this.delegator = delegator;
	}

	@Override
	public void before(String[] events, String[] entities, int order, Handler handler) {
		registerHandler(Phase.BEFORE, EventPredicateTools.create(events, entities), handler, order);
	}

	@Override
	public void on(String[] events, String[] entities, int order, Handler handler) {
		registerHandler(Phase.ON, EventPredicateTools.create(events, entities), handler, order);
	}

	@Override
	public void after(String[] events, String[] entities, int order, Handler handler) {
		registerHandler(Phase.AFTER, EventPredicateTools.create(events, entities), handler, order);
	}

	private void registerHandler(Phase phase, EventPredicate predicate, Handler handler, int order) {
		List<Registration> reg = getRegistration(phase);

		synchronized (reg) {
			int end = reg.size();
			int index = 0;
			while ((index < end) && (order >= reg.get(index).order)) {
				++index;
			}
			reg.add(index, new Registration(predicate, handler, order));
		}
	}

	private List<Registration> getRegistration(Phase phase) {
		switch (phase) {
		case BEFORE:
			return beforeRegistrations; // NOSONAR
		case ON:
			return onRegistrations; // NOSONAR
		case AFTER:
			return afterRegistrations; // NOSONAR
		default:
			throw new ErrorStatusException(CdsErrorStatuses.UNKNOWN_EVENT_PHASE, phase);
		}
	}

	@Override
	public void emit(EventContext context) {
		Objects.requireNonNull(context, "context must not be null");

		// we rely on NameOnlyCdsEntity here
		String entityName = CdsModelUtils.getNameOrEmpty(context.getTarget());
		logger.debug("Started emit of '{}' for event '{}', entity '{}'", getName(), context.getEvent(), entityName);

		Optional<Span> span = OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.EMIT);
		OpenTelemetryUtils.updateSpan(span, getName(), context.getEvent(), entityName);

		try(Scope scope = span.map(Span::makeCurrent).orElse(null)) {

			EventContextSPI innerContext = CdsServiceUtils.getEventContextSPI(context);
			if (innerContext != null) {
				innerContext.setService(delegator);
			}

			// make sure we run 'dispatch' in a ChangeSetContext within a RequestContext and a ChangeSetContext.
			// Create missing context(s) if required.
			boolean hasRequestContext = RequestContext.isActive();
			boolean hasChangeSetContext = ChangeSetContext.isActive();

			if (hasRequestContext && hasChangeSetContext) {
				dispatch(context);

			} else if (!hasRequestContext) {
				new RequestContextRunnerImpl(runtime).run((requestContext) -> {
					if (hasChangeSetContext) {
						dispatch(context);
					} else {
						dispatchInChangeSetContext(context);
					}
				});
			} else {
				dispatchInChangeSetContext(context);
			}
		} catch (Exception e) {
			OpenTelemetryUtils.recordException(span, e);

			throw e;
		} finally {
			OpenTelemetryUtils.endSpan(span);
		}

		logger.debug("Finished emit of '{}' for event '{}', entity '{}'", getName(), context.getEvent(), entityName);
	}

	protected void dispatchInChangeSetContext(EventContext context) {
		// NOTE: lazily creates a ChangeSetContext for a single event (RequestContext is given)
		new ChangeSetContextRunnerImpl(runtime).run((Consumer<ChangeSetContext>) (changeSetContext) -> dispatch(context));
	}

	protected void dispatch(EventContext context) {
		String eventName = context.getEvent();
		String entityName = CdsModelUtils.getNameOrEmpty(context.getTarget());

		if (context.isCompleted()) {
			logger.warn("Tried to dispatch a completed context (service '{}', event '{}', entity '{}')", getName(),
					eventName, entityName);
			return;
		}

		try {
			// BEFORE
			for (Registration registration : beforeRegistrations) {
				if (registration.predicate.test(eventName, entityName)) {
					logHandlerExecution(registration, eventName, entityName, Phase.BEFORE);
					registration.handler.process(context);

					if (context.isCompleted()) {
						break; // end before processing
					}
				}
			}

			// ON
			if (!context.isCompleted()) {
				for (Registration registration : onRegistrations) {
					if (registration.predicate.test(eventName, entityName)) {
						logHandlerExecution(registration, eventName, entityName, Phase.ON);
						registration.handler.process(context);

						if (context.isCompleted()) {
							break; // end on processing
						}
					}
				}
			}

			// AFTER
			for (Registration registration : afterRegistrations) {
				if (registration.predicate.test(eventName, entityName)) {
					logHandlerExecution(registration, eventName, entityName, Phase.AFTER);
					registration.handler.process(context);
				}
			}
		} catch (ContextualizedServiceException e) {
			// add context information
			e.via(context);
			throw e;
		} catch (Exception e) { // NOSONAR
			// wrap exceptions with context information
			throw new ContextualizedServiceException(context, e);
		}
	}

	private void logHandlerExecution(Registration registration, String eventName, String entityName, Phase phase) {
		logger.debug("Executing {} handler '{}' (order {}) for event '{}' on service '{}' and entity '{}'",
				phase, registration.handler, registration.order, eventName, getName(), entityName);
	}

	@Override
	public String getName() {
		return name;
	}

	private static class Registration implements HandlerRegistration {

		public final EventPredicate predicate;
		public final Handler handler;
		public final int order;

		public Registration(EventPredicate predicate, Handler handler, int order) {
			this.predicate = Objects.requireNonNull(predicate, "predicate must not be null");
			this.handler = Objects.requireNonNull(handler, "handler must not be null");
			this.order = order;
		}

		@Override
		public EventPredicate getEventPredicate() {
			return predicate;
		}

		@Override
		public Handler getHandler() {
			return handler;
		}

		@Override
		public int getOrder() {
			return order;
		}

	}

	@Override
	public Stream<HandlerRegistration> registrations(Phase phase) {
		return getRegistration(phase).stream().map(r -> (HandlerRegistration)r);
	}
}
