/**************************************************************************
 * (C) 2021-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.auditlog.DataObject;
import com.sap.cds.services.auditlog.DataSubject;
import com.sap.cds.services.auditlog.KeyValuePair;
import com.sap.cds.services.environment.CdsProperties.AuditLog.PersonalData;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;

/**
 * An abstract base class for personal data analyzer implementations.
 */
abstract class PersonalDataAnalyzer {
	private static final Logger logger = LoggerFactory.getLogger(PersonalDataAnalyzer.class);
	protected final CdsStructuredType entity;
	protected final PersonalDataCaches caches;
	protected final boolean throwOnMissingDataSubject;
	protected List<? extends Map<String, Object>> data;
	protected final PersonalData personalData;

	protected PersonalDataAnalyzer(CdsStructuredType entity, PersonalDataCaches caches, CdsRuntime runtime) {
		this.entity = Objects.requireNonNull(entity, "entity must not be null");
		this.caches = Objects.requireNonNull(caches, "caches must not be null");
		this.throwOnMissingDataSubject = runtime.getEnvironment().getCdsProperties().getAuditLog().getPersonalData()
				.isThrowOnMissingDataSubject();
		this.personalData = runtime.getEnvironment().getCdsProperties().getAuditLog().getPersonalData();
	}

	/**
	 * Indicates whether the given {@link CdsEntity entity} contains personal data or not.
	 *
	 * @return <code>true</code> if given {@link CdsEntity entity} contains personal data.
	 */
	abstract boolean hasPersonalData();

	/**
	 * Sets the {@link List} with new or updated data for current entity.
	 *
	 * @param data a {@link List} with new data
	 */
	public final void setData(List<? extends Map<String, Object>> data) {
		this.data = data;
	}

	protected DataObject createDataObject(Map<String, Object> dataRow, PersonalDataMeta meta) {
		DataObject dataObject = DataObject.create();
		dataObject.setId(createIds(dataRow, meta));
		dataObject.setType(meta.getKey());
		return dataObject;
	}

	protected DataSubject createDataSubject(Map<String, Object> dataRow, PersonalDataMeta meta) {
		PersonalDataMeta dsMeta = findDataSubjectMeta(meta);
		if (dsMeta != null) {
			DataSubject dataSubject = DataSubject.create();
			dataSubject.setRole(dsMeta.getDataSubjectRole());
			dataSubject.setType(dsMeta.getKey());
			List<KeyValuePair> dsIds = createDataSubjectIds(dataRow, dsMeta);
			if (dsIds.isEmpty()) {
				if (this.throwOnMissingDataSubject) {
					throw new ServiceException(CdsErrorStatuses.AUDITLOG_DATA_SUBJECT_MISSING, meta.getKey());
				} else {
					logger.warn("No data subject instance was found.");
				}
			}
			dataSubject.setId(dsIds);
			return dataSubject;
		} else {
			if (this.throwOnMissingDataSubject) {
				throw new ServiceException(CdsErrorStatuses.AUDITLOG_DATA_SUBJECT_MISSING, meta.getKey());
			} else {
				logger.warn("No data subject entity was found for data object {}.", meta.getKey());
			}
		}
		return null;
	}

	private List<KeyValuePair> createIds(Map<String, Object> dataRow, PersonalDataMeta meta) {
		return createIds(dataRow, meta.getKeyNames().stream(), PersonalDataUtils::getKeyAlias);
	}

	private List<KeyValuePair> createDataSubjectIds(Map<String, Object> dataRow, PersonalDataMeta meta) {
		return createIds(dataRow, meta.getDataSubjectIds().stream().map(CdsElement::getName), PersonalDataUtils::getDSIdAlias);
	}

	private List<KeyValuePair> createIds(Map<String, Object> dataRow, Stream<String> keys, UnaryOperator<String> alias) {
		List<KeyValuePair> ids = new ArrayList<>();
		keys.forEach(key -> {
			KeyValuePair pair = KeyValuePair.create();
			pair.setKeyName(key);
			Object keyValue = dataRow.get(alias.apply(key));
			if (keyValue == null) {
				keyValue = dataRow.get(key);
			}
			pair.setValue(keyValue != null ? keyValue.toString() : null);
			ids.add(pair);
		});
		return ids;
	}

	private PersonalDataMeta findDataSubjectMeta(PersonalDataMeta meta) {
		// if given entity is already data subject, stop here and return it
		if (meta.isDataSubject()) {
			return meta;
		} else if (meta.isDataSubjectDetails() || meta.isOther()) {
			// follow association going upwards to data subject
			CdsElement dsAssoc = meta.getDataSubjectAssociation();

			if (dsAssoc != null) {
				CdsEntity parent = dsAssoc.getType().as(CdsAssociationType.class).getTarget();
				return findDataSubjectMeta(getMeta(parent));
			}
		}
		// no data subject available
		return null;
	}

	// cache access

	protected PersonalDataMeta getMeta(CdsStructuredType type) {
		return this.caches.getMeta(type);
	}
}
