/*********************************************************************
 * (C) 2023 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.services.impl.draft;

import static java.util.stream.Collectors.toList;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.CdsDataProcessor.Converter;
import com.sap.cds.Result;
import com.sap.cds.ResultBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Delete;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.impl.SelectBuilder;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.ActiveReadEventContext;
import com.sap.cds.services.draft.DraftService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CqnUtils;

/**
 * Event handlers for events on active versions of draft-enabled entities.
 */
@ServiceName(value = "*", type = DraftService.class)
public class ActiveHandler implements EventHandler {

	@Before(event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE})
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.FILTER_FIELDS)
	protected void clearDraftFields(EventContext context) {
		// draft fields should not be set manually
		CdsServiceUtils.getEntities(context).forEach(r -> {
			CdsModelUtils.visitDeep(context.getTarget(), r, (cdsEntity, data, parent, parentData) -> {
				data.remove(Drafts.HAS_DRAFT_ENTITY);
				data.remove(Drafts.HAS_ACTIVE_ENTITY);
				data.remove(Drafts.SIBLING_ENTITY);
				data.remove(Drafts.DRAFT_ADMINISTRATIVE_DATA);
				data.remove(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID);
				if (parentData != null && !data.values().stream().anyMatch(e -> e != null)) {
					parentData.remove(parent.getName());
				}
			});
		});
	}

	@Before(event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE})
	// IsActiveEntity must be cleared late, as it is part of the key and relevant for validation targets
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.VALIDATE_FIELDS)
	protected void clearIsActiveEntity(EventContext context) {
		CdsDataProcessor.create()
				.addConverter((p, e, t) -> e.getName().equals(Drafts.IS_ACTIVE_ENTITY), (p, e, v) -> Converter.REMOVE)
				.process(CdsServiceUtils.getEntities(context), context.getTarget());
	}

	@Before(service = "*", serviceType = PersistenceService.class, event = {CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE})
	@HandlerOrder(HandlerOrder.AFTER + OrderConstants.Before.FILTER_FIELDS)
	protected void clearDraftFieldsOnPersistenceService(EventContext context) {
		if(DraftUtils.isDraftEnabled(context.getTarget()) && !context.getTarget().getQualifiedName().endsWith(DraftModifier.DRAFT_SUFFIX)) {
			clearDraftFields(context);
		}
	}

	@After(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPSERT, CqnService.EVENT_UPDATE })
	@HandlerOrder(OrderConstants.After.ADD_FIELDS)
	@SuppressWarnings("unchecked")
	protected void addActiveDraftFields(EventContext context) {
		List<?> result = CdsServiceUtils.getResult(context).list();
		CdsModelUtils.visitDeep(context.getTarget(), (List<Map<String, Object>>) result, (cdsEntity, data, parent) -> {
			if(DraftUtils.isDraftEnabled(cdsEntity)) {
				data.forEach(m -> {
					m.put(Drafts.IS_ACTIVE_ENTITY, true);
					m.put(Drafts.HAS_ACTIVE_ENTITY, false);
					m.put(Drafts.HAS_DRAFT_ENTITY, false);
				});
			}
		});
	}

	@Before(event = {DraftService.EVENT_ACTIVE_READ, CqnService.EVENT_UPDATE, CqnService.EVENT_DELETE})
	@HandlerOrder(OrderConstants.Before.ADAPT_STATEMENT)
	protected void adaptForActiveExecution(EventContext context) {
		// TODO only required for specific ACTIVE_READ events?
		if (context.getEvent().equals(DraftService.EVENT_ACTIVE_READ) || DraftUtils.isDraftEnabled(context.getTarget())) {
			CqnStatement statement = (CqnStatement) context.get("cqn");
			if (!statement.isSelect()) {
				statement = addLockingConstraints(statement, context);
			}
			statement = CqnAdapter.create(context).adaptForActiveExecution(statement);
			if (statement == null || (!statement.isSelect() && isTargetingLocked(statement, context))) {
				context.put("result", DraftHandlerUtils.buildNoOpResult(context));
				context.setCompleted();
			} else {
				// target entity of context might not exactly fit any more
				context.put("cqn", statement);
			}
		}
	}

	@Before
	@HandlerOrder(OrderConstants.Before.ADAPT_STATEMENT + 1)
	protected void selectRelatedDraftsBeforeDelete(CdsDeleteEventContext context) {
		if (DraftUtils.isDraftEnabled(context.getTarget())) {
			context.put("relatedDraftUuids", selectRelatedDraftUuids(context.getCqn(), context));
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultRead(ActiveReadEventContext context) {
		return CdsServiceUtils.getDefaultPersistenceService(context).run(context.getCqn(), context.getCqnNamedValues());
	}

	@After
	@SuppressWarnings("unchecked")
	protected void deleteRelatedDrafts(CdsDeleteEventContext context) {
		if (DraftUtils.isDraftEnabled(context.getTarget())) {
			// If draft keys were selected, run the statement with these keys unless they are empty.
			// Otherwise, the `cancelDraft` will receive the statement and will execute it on drafts only.
			if (context.getResult().rowCount() > 0) {
				DraftService service = (DraftService) context.getService();
				List<List<?>> draftKeysToDelete = (List<List<?>>) context.get("relatedDraftUuids");
				context.put("relatedDraftUuids", null);
				Result deletedDrafts = context.getCdsRuntime().requestContext().privilegedUser().run(requestContext -> {
					if (draftKeysToDelete != null) {
						// Assume that in() predicate will not create additional batches and the number of batches
						// will match the number of them in the result of the delete statement for active entities
						long[] rowCount = draftKeysToDelete.stream()
							.mapToLong(item -> {
								if (!item.isEmpty()) {
									return service.cancelDraft(
										Delete.from(context.getTarget())
											.where(c -> c.get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID).in(item))).rowCount();
								}
								return 0L;
							}).toArray();
						return ResultBuilder.deletedRows(rowCount).result();
					} else {
						return service.cancelDraft(context.getCqn(), context.getCqnValueSets());
					}
				});
				context.setResult(ResultBuilder.deletedRows(addRowCounts(context.getResult(), deletedDrafts)).result());
			}
		}
	}

	// used in context of joint persistence, adds a not exists subquery condition to the statement
	private <T extends CqnStatement> T addLockingConstraints(T statement, EventContext context) {
		CdsProperties properties = context.getCdsRuntime().getEnvironment().getCdsProperties();
		if (!properties.getSecurity().getAuthorization().getDraftProtection().isEnabled() || "split".equals(DraftHandlerUtils.getDraftPersistenceMode(context))) {
			return statement;
		}
		CdsEntity draftEntity = DraftModifier.getDraftsEntity(context.getTarget(), context.getModel());
		CqnPredicate sameKeys = context.getTarget().keyElements()
				// TODO Remove IS_ACTIVE_ENTITY if we make it virtual again
				.filter(e -> !e.getType().isAssociation() && !e.isVirtual() && !e.getName().equals(Drafts.IS_ACTIVE_ENTITY))
				.map(key -> CQL.to(CqnExistsSubquery.OUTER).get(key.getName()).eq(CQL.get(key.getName())))
				.collect(CQL.withAnd());

		// update: actives with any draft are locked for update
		if (statement.isDelete()) {
			// deletion: only actives with locked drafts by another user are locked for deletion
			sameKeys = CQL.and(sameKeys, DraftActivesReader.ownedbyAnotherUser(true, context));
		}
		return CqnUtils.addWhere(statement, CQL.exists(Select.from(draftEntity).where(sameKeys)).not());
	}

	// used in context of split persistence, explicitly checks the lock by running a query
	private boolean isTargetingLocked(CqnStatement statement, EventContext context) {
		CdsProperties properties = context.getCdsRuntime().getEnvironment().getCdsProperties();
		if (!properties.getSecurity().getAuthorization().getDraftProtection().isEnabled() || !"split".equals(DraftHandlerUtils.getDraftPersistenceMode(context))) {
			return false;
		}

		Select<?> select;
		Iterable<Map<String, Object>> valueSets;
		if (statement.isUpdate()) {
			// update: actives with any draft are locked for update
			select = CqnUtils.toSelect(statement.asUpdate(), context.getTarget());
			valueSets = context.as(CdsUpdateEventContext.class).getCqnValueSets();
		} else if (statement.isDelete()) {
			// deletion: only actives with locked drafts by another user are locked for deletion
			select = CqnUtils.addWhere(CqnUtils.toSelect(statement.asDelete()), DraftActivesReader.ownedbyAnotherUser(true, context));
			valueSets = context.as(CdsDeleteEventContext.class).getCqnValueSets();
		} else {
			return false;
		}

		CqnSelect draftExists = select.columns(CQL.constant(1).as("one")).limit(1);
		Iterator<Map<String, Object>> iterator = valueSets.iterator();
		if (!iterator.hasNext()) {
			return DraftInactivesReader.create(context).draftsOfAllUsers(draftExists).rowCount() > 0;
		} else {
			while (iterator.hasNext()) {
				if (DraftInactivesReader.create(context).draftsOfAllUsers(draftExists, iterator.next()).rowCount() > 0) {
					return true; // if one target is locked, the statement is not executed
				}
			}
			return false;
		}
	}

	private long[] addRowCounts(Result... results) {
		// important assumption: all results that come here must have at least the same
		// number of batches as the first one

		long[] result = new long[0];
		for (Result r: results) {
			if (r != null) {
				if (result.length == 0) {
					result = new long[r.batchCount()];
				}
				for (int i = 0; i < result.length; ++i) {
					result[i] += r.rowCount(i);
				}
			}
		}
		return result;
	}

	private List<List<?>> selectRelatedDraftUuids(CqnDelete delete, CdsDeleteEventContext context) {
		// We need all IDs of the drafts that are matching the condition of Delete statement that is passed to
		// this event as an argument. If condition is not present in where or in ref(), nothing will be selected.
		// TODO consider if reading DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID via readDraft is also possible
		if (delete.where().isPresent() || delete.ref().targetSegment().filter().isPresent()) {
			CqnSelect select = delete.where()
					.map(condition -> selectDraftID(delete.ref()).where(w -> condition))
					.orElse(selectDraftID(delete.ref()))
					.filter(CQL.get(Drafts.IS_ACTIVE_ENTITY).eq(true));

			if (context.getCqnValueSets().iterator().hasNext()) {
				List<List<?>> result = new ArrayList<>();
				context.getCqnValueSets().forEach(parameters ->
					result.add(context.getService().run(select, parameters).stream()
						.map(row -> row.get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
						.filter(Objects::nonNull).collect(toList())));
				return result;
			}
			return Collections.singletonList(context.getService().run(select).stream()
				.map(row -> row.get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
				.filter(Objects::nonNull).collect(toList()));
		}
		return null;
	}

	private static SelectBuilder<StructuredType<?>> selectDraftID(CqnStructuredTypeRef ref) {
		return SelectBuilder.from(ref).columns(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID);
	}

}
