package com.sap.cds.services.impl.draft;

import java.util.HashMap;
import java.util.Map;

import com.sap.cds.Result;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.services.EventContext;

/**
 * Used in {@link DraftActionsHandler#defaultRead(com.sap.cds.services.cds.CdsReadEventContext)} to handle reading of sanitized inactives.
 */
class DraftInactivesReader {

	public static DraftInactivesReader create(EventContext context) {
		return new DraftInactivesReader(context);
	}

	private final EventContext context;

	private DraftInactivesReader(EventContext context) {
		this.context = context;
	}

	public Result allInactives(CqnSelect select, Map<String, Object> cqnNamedValues, CqnPredicate remainingWhere, CqnStructuredTypeRef remainingRef, boolean draftAdminFromActive) {
		CqnSelect sanitized = CQL.copy(select, new InactiveSanitizer(remainingWhere, remainingRef));
		if (draftAdminFromActive) {
			return draftsOfAllUsers(sanitized, cqnNamedValues);
		}
		return DraftServiceImpl.downcast(context.getService()).readDraft(sanitized, cqnNamedValues);
	}

	public Result draftsOfAllUsers(CqnSelect select) {
		return draftsOfAllUsers(select, new HashMap<>(0));
	}

	public Result draftsOfAllUsers(CqnSelect select, Map<String, Object> cqnNamedValues) {
		// drafts of other users need to be found as well
		// draft metadata is public and visible to everyone
		// however some fields in draft administrative data take the user name into account
		return context.getCdsRuntime().requestContext().privilegedUser().modifyUser(u -> u.setName(context.getUserInfo().getName())).run(requestContext -> {
			return DraftServiceImpl.downcast(context.getService()).readDraft(select, cqnNamedValues);
		});
	}

	private static class InactiveSanitizer implements Modifier {

		private final CqnPredicate remainingWhere;
		private final CqnStructuredTypeRef remainingRef;

		public InactiveSanitizer(CqnPredicate remainingWhere, CqnStructuredTypeRef remainingRef) {
			this.remainingWhere = remainingWhere;
			this.remainingRef = remainingRef;
		}

		@Override
		public CqnStructuredTypeRef ref(CqnStructuredTypeRef ref) {
			return remainingRef;
		}

		@Override
		public CqnPredicate where(Predicate where) {
			return remainingWhere;
		}

	}

}
