package com.sap.cds.services.impl.draft;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.impl.util.Stack;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.RefBuilder;
import com.sap.cds.ql.RefBuilder.RefSegment;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnArithmeticExpression;
import com.sap.cds.ql.cqn.CqnArithmeticNegation;
import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnComparisonPredicate.Operator;
import com.sap.cds.ql.cqn.CqnConnectivePredicate;
import com.sap.cds.ql.cqn.CqnContainmentTest;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnExpression;
import com.sap.cds.ql.cqn.CqnFunc;
import com.sap.cds.ql.cqn.CqnInPredicate;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnListValue;
import com.sap.cds.ql.cqn.CqnLiteral;
import com.sap.cds.ql.cqn.CqnMatchPredicate;
import com.sap.cds.ql.cqn.CqnNegation;
import com.sap.cds.ql.cqn.CqnNullValue;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnPlain;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnSearchPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnToken;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.ql.impl.Xpr;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.DraftUtils;
import com.sap.cds.util.CdsModelUtils;

public class DraftScenarioAnalyzer {

	public record AnalysisResult (Scenario scenario, CqnPredicate predicate, CqnStructuredTypeRef ref) {}

	public static AnalysisResult analyze(CqnSelect select, CdsEntity target, CdsModel model) {
		if ((DraftUtils.isDraftEnabled(target) || target.getName().equals(Drafts.DRAFT_ADMINISTRATIVE_DATA)) &&
				select.groupBy().isEmpty() && select.having().isEmpty() && select.from().isRef() &&
				StatementAnalyzer.analyze(select, target).isOptimizable() &&
				PredicateAnalyzer.analyze(select.search()).getComparison() == Comparison.OTHER) {

			CqnStructuredTypeRef ref = select.ref();
			RefAnalyzer refAnalyzer = RefAnalyzer.analyze(ref, model);
			PredicateAnalyzer whereAnalyzer = PredicateAnalyzer.analyze(select.where());

			Scenario scenario = mergeScenarios(refAnalyzer.getScenario(), whereAnalyzer.getScenario(), ref);
			return new AnalysisResult(scenario, whereAnalyzer.getPredicate(), refAnalyzer.getRemainingRef());
		}
		return new AnalysisResult(Scenario.NOT_OPTIMIZABLE, null, null);
	}

	private static Scenario mergeScenarios(Scenario refScenario, Scenario whereScenario, CqnStructuredTypeRef ref) {
		Scenario scenario;
		if (refScenario == Scenario.OTHER) {
			scenario = whereScenario;
		} else if ((refScenario == Scenario.ALL_HIDING_DRAFTS && (whereScenario == Scenario.ALL_HIDING_DRAFTS || whereScenario == Scenario.OTHER)) ||
				(refScenario == Scenario.OWN_DRAFT && (whereScenario == Scenario.OWN_DRAFT || whereScenario == Scenario.OTHER))) {
			String lastSegment = ref.lastSegment();
			if (lastSegment.equals(Drafts.SIBLING_ENTITY)) {
				// flip scenario in case of SiblingEntity navigation
				scenario = refScenario == Scenario.OWN_DRAFT ? Scenario.ALL_HIDING_DRAFTS : Scenario.OWN_DRAFT;
			} else if (lastSegment.equals(Drafts.DRAFT_ADMINISTRATIVE_DATA)) {
				// always read DraftAdministrativeData via draft
				scenario = refScenario == Scenario.OWN_DRAFT ? Scenario.OWN_DRAFT : Scenario.DRAFT_ADMINISTRATIVE_DATA_VIA_ACTIVE;
			} else {
				scenario = refScenario;
			}
		} else {
			scenario = Scenario.NOT_OPTIMIZABLE;
		}
		return scenario;
	}

	private static class StatementAnalyzer implements CqnVisitor {

		private static StatementAnalyzer analyze(CqnSelect select, CdsEntity target) {
			StatementAnalyzer validator = new StatementAnalyzer(target, false);
			select.accept(validator);
			return validator;
		}

		private final CdsEntity entity;
		private final Set<String> keyNames;
		private final boolean forExpand;

		private boolean optimizable = true;

		public StatementAnalyzer(CdsEntity entity, boolean forExpand) {
			this.entity = entity;
			this.keyNames = entity.keyElements().map(CdsElement::getName).collect(Collectors.toSet());
			this.forExpand = forExpand;
		}

		public boolean isOptimizable() {
			return optimizable;
		}


		@Override
		public void visit(CqnStructuredTypeRef typeRef) {
			if (optimizable && (
					(!forExpand && typeRef.segments().subList(0, typeRef.size() - 1).stream().anyMatch(s -> s.id().equals(Drafts.SIBLING_ENTITY))) || // sibling entity navigation only at the very end
					(forExpand && (typeRef.size() != 1 || typeRef.rootSegment().filter().isPresent() || typeRef.rootSegment().id().equals(Drafts.SIBLING_ENTITY))) // no multi segment expands or filters in expands and no SiblingEntity expand
				)) {
				optimizable = false;
			}
		}

		@Override
		public void visit(CqnSortSpecification sortSpec) {
			// orderby must not contain draft elements, except IsActiveEntity
			if(optimizable && sortSpec.value().ofRef()
								.filter(r -> !r.path().equals(Drafts.IS_ACTIVE_ENTITY))
								.anyMatch(DraftScenarioAnalyzer::containsDraftElements)) {
				optimizable = false;
			}
		}

		@Override
		public void visit(CqnSelectListValue slv) {
			if (optimizable) {
				CqnValue value = slv.value();
				String elementName = null;
				if (value.isRef()) {
					// no multi-segment references to draft elements
					CqnElementRef ref = value.asRef();
					if (ref.size() > 1 && containsDraftElements(ref)) {
						optimizable = false;
					} else {
						elementName = ref.path();
					}
				} else if (value.isFunction()) {
					if (value.asFunction().args().stream().flatMap(v -> v.ofRef()).anyMatch(DraftScenarioAnalyzer::containsDraftElements)) {
						optimizable = false;
					}
				} else if (refsOf(value).anyMatch(DraftScenarioAnalyzer::containsDraftElements)) {
					optimizable = false;
				}

				if (optimizable && slv.alias().isPresent() && !slv.alias().get().equals(elementName) &&
					(keyNames.contains(slv.alias().get()) || (elementName != null && Drafts.ELEMENTS.contains(elementName)))) {
					optimizable = false;
				}
			}
		}

		@Override
		public void visit(CqnExpand expand) {
			if (optimizable) {
				if (expand.ref().path().equals("*")) {
					optimizable = false;
				} else {
					StatementAnalyzer analyzer = new StatementAnalyzer(CdsModelUtils.entity(entity, expand.ref().segments()), true);
					expand.ref().accept(analyzer);
					expand.items().forEach(i -> i.accept(analyzer));
					expand.orderBy().forEach(o -> o.accept(analyzer));
					optimizable = analyzer.isOptimizable();
				}
			}
		}

		@Override
		public void visit(CqnInline inline) {
			optimizable = false;
		}

		private Stream<CqnElementRef> refsOf(CqnToken token) {
			return token.tokens().filter(CqnElementRef.class::isInstance).map(CqnElementRef.class::cast);
		}

	}

	public enum Scenario {
		ALL,
		ALL_HIDING_DRAFTS,
		UNCHANGED,
		OWN_DRAFT,
		LOCKED_BY_ANOTHER_USER,
		UNSAVED_CHANGES_BY_ANOTHER_USER,
		DRAFT_ADMINISTRATIVE_DATA_VIA_ACTIVE, // no Fiori draft scenario, internal marker
		OTHER, // no Fiori draft scenario, but optimizable
		NOT_OPTIMIZABLE
	}

	private enum Comparison {
		IS_ACTIVE_ENTITY_EQ_TRUE,
		IS_ACTIVE_ENTITY_EQ_FALSE,
		SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL,
		HAS_DRAFT_ENTITY_EQ_FALSE,
		DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_EQ_EMPTY,
		DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_EMPTY,
		DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_NULL,
		DRAFT_RELATED,
		OTHER
	}

	private static class PredicateAnalyzer implements CqnVisitor {

		private static PredicateAnalyzer analyze(Optional<CqnPredicate> predicate) {
			PredicateAnalyzer analyzer = new PredicateAnalyzer();
			predicate.ifPresent(p -> p.accept(analyzer));
			return analyzer;
		}

		private static final String IS_ACTIVE_ENTITY = Drafts.IS_ACTIVE_ENTITY;
		private static final String SIBLING_ENTITY_IS_ACTIVE_ENTITY = Drafts.SIBLING_ENTITY + "." + Drafts.IS_ACTIVE_ENTITY;
		private static final String HAS_DRAFT_ENTITY = Drafts.HAS_DRAFT_ENTITY;
		private static final String DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER = Drafts.DRAFT_ADMINISTRATIVE_DATA + "." + DraftAdministrativeData.IN_PROCESS_BY_USER;

		private final Stack<Object> scenarioStack = new Stack<>();
		private final Stack<CqnToken> predicateStack = new Stack<>();

		public Scenario getScenario() {
			Object comparison = getComparison();
			if (comparison == Comparison.IS_ACTIVE_ENTITY_EQ_TRUE) {
				comparison = Scenario.ALL_HIDING_DRAFTS;
			} else if (comparison == Comparison.IS_ACTIVE_ENTITY_EQ_FALSE) {
				comparison = Scenario.OWN_DRAFT;
			} else if (comparison == Comparison.OTHER) {
				comparison = Scenario.OTHER;
			}
			return comparison instanceof Scenario s ? s : Scenario.NOT_OPTIMIZABLE;
		}

		public CqnPredicate getPredicate() {
			CqnToken token = predicateStack.size() == 1 ? predicateStack.pop() : CQL.TRUE;
			return token instanceof CqnPredicate p ? p : CQL.TRUE;
		}

		private Object getComparison() {
			return scenarioStack.size() == 1 ? scenarioStack.pop() : Comparison.OTHER;
		}

		@Override
		public void visit(CqnComparisonPredicate c) {
			Object comparison = comparisonType(scenarioStack.pop(2));
			String ref = c.left().isRef() ? c.left().asRef().path() :
						c.right().isRef() ? c.right().asRef().path() : null;
			// one side must be a ref
			if (ref != null) {
				boolean isNull = c.left().isNullValue() || c.right().isNullValue();
				Object value = c.left().isLiteral() ? c.left().asLiteral().value() :
							c.right().isLiteral() ? c.right().asLiteral().value() : null;
				// compared with null or some literal value
				if (isNull || value != null) {
					var op = c.operator();
					if (ref.equals(IS_ACTIVE_ENTITY) && (op == Operator.IS || op == Operator.EQ)) {
						if (Boolean.TRUE.equals(value)) {
							comparison = Comparison.IS_ACTIVE_ENTITY_EQ_TRUE;
						} else if (Boolean.FALSE.equals(value)) {
							comparison = Comparison.IS_ACTIVE_ENTITY_EQ_FALSE;
						}
					} else if (ref.equals(SIBLING_ENTITY_IS_ACTIVE_ENTITY) && op == Operator.IS && isNull) {
						comparison = Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL;
					} else if (ref.equals(HAS_DRAFT_ENTITY) && (op == Operator.IS || op == Operator.EQ) && Boolean.FALSE.equals(value)) {
						comparison = Comparison.HAS_DRAFT_ENTITY_EQ_FALSE;
					} else if (ref.equals(DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER)) {
						if ("".equals(value)) {
							if (op == Operator.IS || op == Operator.EQ) {
								comparison = Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_EQ_EMPTY;
							} else if (op == Operator.IS_NOT || op == Operator.NE) {
								comparison = Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_EMPTY;
							}
						} else if (op == Operator.IS_NOT && isNull) {
							comparison = Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_NULL;
						}
					}
				}
			}
			scenarioStack.push(comparison);
			predicateStack.pop(2);
			// replace scenario comparisons with TRUE
			if (comparison == Comparison.OTHER || comparison == Comparison.DRAFT_RELATED) {
				predicateStack.push(c);
			} else {
				predicateStack.push(CQL.TRUE);
			}
		}

		@Override
		public void visit(CqnConnectivePredicate connective) {
			Set<Object> comparisons = new HashSet<>(scenarioStack.pop(connective.predicates().size()));
			// a draft scenario connected with AND with other non draft scenarios or comparisons, remains the draft scenario
			// therefore we can ignore Comparison.OTHER if the Operator is AND
			if (connective.operator() == CqnConnectivePredicate.Operator.AND) {
				comparisons.remove(Comparison.OTHER);
			}
			if (comparisons.size() == 2 && connective.operator() == CqnConnectivePredicate.Operator.OR &&
					comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_FALSE) &&
					comparisons.contains(Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL)) {
				scenarioStack.push(Scenario.ALL);
			} else if (comparisons.size() == 2 && connective.operator() == CqnConnectivePredicate.Operator.AND &&
					comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE) &&
					comparisons.contains(Comparison.HAS_DRAFT_ENTITY_EQ_FALSE)) {
				scenarioStack.push(Scenario.UNCHANGED);
			} else if (comparisons.size() == 3 && connective.operator() == CqnConnectivePredicate.Operator.AND &&
					comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE) &&
					comparisons.contains(Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL) &&
					comparisons.contains(Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_EQ_EMPTY)) {
				scenarioStack.push(Scenario.UNSAVED_CHANGES_BY_ANOTHER_USER);
			} else if (comparisons.size() == 4 && connective.operator() == CqnConnectivePredicate.Operator.AND &&
					comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE) &&
					comparisons.contains(Comparison.SIBLING_ENTITY_IS_ACTIVE_ENTITY_EQ_NULL) &&
					comparisons.contains(Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_EMPTY) &&
					comparisons.contains(Comparison.DRAFT_ADMINISTRATIVE_DATA_IN_PROCESS_BY_USER_NE_NULL)) {
				scenarioStack.push(Scenario.LOCKED_BY_ANOTHER_USER);
			} else if (comparisons.size() == 1 && connective.operator() == CqnConnectivePredicate.Operator.AND &&
					(comparisons.iterator().next() instanceof Scenario ||
					comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_TRUE) ||
					comparisons.contains(Comparison.IS_ACTIVE_ENTITY_EQ_FALSE))) {
				scenarioStack.push(comparisons.iterator().next());
			} else {
				scenarioStack.push(comparisonType(comparisons));
			}
			// merge potentially replaced predicates
			List<CqnToken> predicateTokens = predicateStack.pop(connective.predicates().size());
			predicateStack.push(CQL.connect(connective.operator(), predicateTokens.stream().map(t -> (CqnPredicate) t).toList()));
		}

		@Override
		public void visit(CqnElementRef elementRef) {
			if (containsDraftElements(elementRef)) {
				scenarioStack.push(Comparison.DRAFT_RELATED);
			} else {
				scenarioStack.push(Comparison.OTHER);
			}
			predicateStack.push(elementRef);
		}

		@Override
		public void visit(CqnPlain plain) {
			scenarioStack.push(Comparison.DRAFT_RELATED); // potentially
			predicateStack.push(plain);
		}

		@Override
		public void visit(CqnParameter param) {
			pushOther(param);
		}

		@Override
		public void visit(CqnLiteral<?> literal) {
			pushOther(literal);
		}

		@Override
		public void visit(CqnNullValue nil) {
			pushOther(nil);
		}

		@Override
		public void visit(CqnExistsSubquery exists) {
			pushOther(exists);
		}

		@Override
		public void visit(CqnMatchPredicate match) {
			pushOther(match);
		}

		@Override
		public void visit(CqnFunc func) {
			popPush(func.args().size(), func);
		}

		@Override
		public void visit(CqnListValue list) {
			popPush(list.size(), list);
		}

		@Override
		public void visit(CqnSearchPredicate search) {
			popPush(1, search);
		}

		@Override
		public void visit(CqnContainmentTest test) {
			popPush(2, test);
		}

		@Override
		public void visit(CqnInPredicate in) {
			popPush(2, in);
		}

		@Override
		public void visit(CqnArithmeticExpression expr) {
			popPush(2, expr);
		}

		@Override
		public void visit(CqnArithmeticNegation neg) {
			popPush(1, neg);
		}

		@Override
		public void visit(CqnNegation neg) {
			popPush(1, neg);
		}

		@Override
		public void visit(CqnExpression expr) {
			popPush(((Xpr) expr).length(), expr);
		}

		private Object comparisonType(Collection<Object> comparisons) {
			return comparisons.stream().allMatch(Comparison.OTHER::equals) ? Comparison.OTHER : Comparison.DRAFT_RELATED;
		}

		private void pushOther(CqnToken token) {
			scenarioStack.push(Comparison.OTHER);
			predicateStack.push(token);
		}

		private void popPush(int pop, CqnToken token) {
			scenarioStack.push(comparisonType(scenarioStack.pop(pop)));
			predicateStack.pop(pop);
			predicateStack.push(token);
		}

	}

	private static boolean containsDraftElements(CqnElementRef ref) {
		return ref.segments().stream().map(Segment::id).anyMatch(Drafts.ELEMENTS::contains);
	}

	private static class RefAnalyzer {

		private final CdsModel model;
		private CqnStructuredTypeRef remainingRef;
		private Scenario scenario = Scenario.NOT_OPTIMIZABLE;

		public static RefAnalyzer analyze(CqnStructuredTypeRef ref, CdsModel model) {
			RefAnalyzer analyzer = new RefAnalyzer(ref, model);
			analyzer.analyze();
			return analyzer;
		}

		private RefAnalyzer(CqnStructuredTypeRef ref, CdsModel model) {
			this.model = model;
			this.remainingRef = ref;
		}

		public Scenario getScenario() {
			return scenario;
		}

		public CqnStructuredTypeRef getRemainingRef() {
			return remainingRef;
		}

		private void analyze() {
			RefBuilder<StructuredTypeRef> ref = CQL.copy(remainingRef);
			PredicateAnalyzer rootAnalyzer = PredicateAnalyzer.analyze(ref.rootSegment().filter());
			Scenario scenario = rootAnalyzer.getScenario();
			if (scenario == Scenario.ALL_HIDING_DRAFTS || scenario == Scenario.OWN_DRAFT) {
				ref.rootSegment().filter(rootAnalyzer.getPredicate());

				int endIndex = ref.segments().size();
				if (ref.targetSegment().id().equals(Drafts.SIBLING_ENTITY)) {
					ref.segments().remove(--endIndex); // remove SIBLING_ENTITY
				} else if (ref.targetSegment().id().equals(Drafts.DRAFT_ADMINISTRATIVE_DATA)) {
					--endIndex; // ignore DRAFT_ADMINISTRATIVE_DATA
				}

				Iterator<ResolvedSegment> iterator = CqnAnalyzer.create(model).analyze(remainingRef).iterator();
				for (int i=1; i<endIndex; ++i) {
					RefSegment segment = ref.segments().get(i);
					CdsEntity parentEntity = iterator.next().entity();

					PredicateAnalyzer segmentAnalyzer = PredicateAnalyzer.analyze(segment.filter());
					Scenario segmentScenario = segmentAnalyzer.getScenario();
					boolean isComposition = parentEntity.findAssociation(segment.id())
							.map(a -> a.getType().as(CdsAssociationType.class).isComposition())
							.orElse(false);
					// we only optimize along compositions that keep the scenario
					// TODO composition is not the right indicator for the same draft document
					// also consider backlink associations and associations with @odata.draft.enclosed
					if (!isComposition || (segmentScenario != scenario && segmentScenario != Scenario.OTHER)) {
						return;
					}
					segment.filter(segmentAnalyzer.getPredicate());
				}

			}
			this.scenario = scenario;
			this.remainingRef = ref.build();
		}

	}

}
