package com.sap.cds.services.impl.outbox.persistence.collectors;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.sap.cds.services.impl.outbox.persistence.PersistentOutbox;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * Coordinator implementation that manages all outbox partition collector
 * threads.
 */
public class PartitionCollectorCoordinator {

	public static final int OUTBOX_MESSAGE_PARTITIONS = 2;
	private static final Logger LOG = LoggerFactory.getLogger(PartitionCollectorCoordinator.class);

	private final CdsRuntime runtime;
	private final int partitions = OUTBOX_MESSAGE_PARTITIONS;

	// +1 thread for tenant info update schedule
	private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(partitions + 1,
			new ThreadFactoryBuilder().setDaemon(true).build());
	private final Map<Integer, PartitionCollector> collectors = new ConcurrentHashMap<>();

	private final TenantProviderService tenantService;
	private volatile List<TenantInfo> tenantsInfo;

	public PartitionCollectorCoordinator(CdsRuntime runtime) {
		this.runtime = runtime;
		this.tenantService = runtime.getServiceCatalog().getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);

		long tenantsInfoRefreshIntervalSeconds = runtime.getEnvironment().getCdsProperties().getOutbox().getPersistent()
				.getTenantsInfoRefreshInterval().getSeconds();
		this.scheduler.scheduleAtFixedRate(this::updateTenants, 0, tenantsInfoRefreshIntervalSeconds, TimeUnit.SECONDS);
	}

	private void updateTenants() {
		boolean initial = tenantsInfo == null;
		try {
			LOG.debug("Updating cached tenants for partition collectors");
			tenantsInfo = Collections.unmodifiableList(this.tenantService.readTenantsInfo());
			if(initial) {
				initCollectors();
			}
		} catch (Exception e) {
			LOG.warn("Failed to update cached tenants for partition collectors", e);
		}
	}

	private void initCollectors() {
		LOG.info("Initializing collectors for {} partitions", this.partitions);
		PersistentOutbox outboxService = (PersistentOutbox) runtime.getServiceCatalog().getService(OutboxService.class,
				OutboxService.PERSISTENT_NAME);
		for (int partition = 0; partition < this.partitions; ++partition) {
			PartitionCollector collector = new PartitionCollector(runtime, outboxService, () -> tenantsInfo, partition);
			this.collectors.put(partition, collector);
			scheduler.execute(collector);
		}
	}

	public void schedule(int partition) {
		PartitionCollector collector = this.collectors.get(partition);
		if (collector != null) {
			collector.unpause();
		}
	}

	public void stop() {
		// will interrupt PartitionCollector
		scheduler.shutdownNow();
	}

}
