/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.reflect.TypeToken;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.Service;
import com.sap.cds.services.authentication.AuthenticationInfo;
import com.sap.cds.services.environment.ApplicationInfo;
import com.sap.cds.services.environment.ApplicationInfoProvider;
import com.sap.cds.services.environment.CdsEnvironment;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.environment.PropertiesProvider;
import com.sap.cds.services.environment.ServiceBindingProvider;
import com.sap.cds.services.impl.ServiceCatalogImpl;
import com.sap.cds.services.impl.ServiceCatalogSPI;
import com.sap.cds.services.impl.ServiceExceptionUtilsImpl;
import com.sap.cds.services.impl.ServiceSPI;
import com.sap.cds.services.impl.environment.DefaultApplicationInfoProvider;
import com.sap.cds.services.impl.environment.DefaultServiceBindingProvider;
import com.sap.cds.services.impl.environment.SimplePropertiesProvider;
import com.sap.cds.services.impl.handlerregistry.HandlerRegistryTools;
import com.sap.cds.services.impl.messages.SimpleLocalizedMessageProvider;
import com.sap.cds.services.impl.runtime.mockusers.MockedFeatureTogglesProvider;
import com.sap.cds.services.impl.runtime.mockusers.MockedUserInfoProvider;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.messages.LocalizedMessageProvider;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.AuthenticationInfoProvider;
import com.sap.cds.services.runtime.CdsModelProvider;
import com.sap.cds.services.runtime.CdsProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.ChangeSetContextRunner;
import com.sap.cds.services.runtime.FeatureTogglesInfoProvider;
import com.sap.cds.services.runtime.ParameterInfoProvider;
import com.sap.cds.services.runtime.RequestContextRunner;
import com.sap.cds.services.runtime.UserInfoProvider;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

/**
 * Bootstrap for setting up service backbone. This includes reading the CDS
 * model, creating corresponding services and registering default and custom
 * handlers.
 */
public class CdsRuntimeImpl implements CdsRuntime {

	private static final Logger logger = LoggerFactory.getLogger(CdsRuntimeImpl.class);
	private static final String BASE_PACKAGE = "com.sap.cds.";

	private final CdsEnvironment environment;
	private final ServiceCatalogImpl serviceCatalog = new ServiceCatalogImpl();
	private final Map<Class<? extends CdsProvider<?>>, CdsProvider<?>> providers = new HashMap<>();

	private CdsModel cdsModel = CdsModelUtils.loadCdsModel(null); // empty model

	// only visible in this package
	CdsRuntimeImpl(PropertiesProvider propertiesProvider) {
		this.environment = new CdsEnvironmentImpl(propertiesProvider);

		// initialize default providers
		providers.put(ServiceBindingProvider.class, new DefaultServiceBindingProvider(this));
		providers.put(ApplicationInfoProvider.class, new DefaultApplicationInfoProvider(this));
		providers.put(CdsModelProvider.class, new DefaultModelProvider());
		providers.put(UserInfoProvider.class, new MockedUserInfoProvider(this)); // will be overriden by authentication features
		providers.put(AuthenticationInfoProvider.class, (AuthenticationInfoProvider) () -> null); // needs to be set by the app framework
		providers.put(ParameterInfoProvider.class, (ParameterInfoProvider) () -> null); // needs to be set by app framework
		providers.put(LocalizedMessageProvider.class, new SimpleLocalizedMessageProvider());
		providers.put(FeatureTogglesInfoProvider.class, new MockedFeatureTogglesProvider(this));
	}

	@Override
	public ServiceCatalogSPI getServiceCatalog() {
		return serviceCatalog;
	}

	@Override
	public CdsModel getCdsModel() {
		return cdsModel;
	}

	@Override
	public CdsModel getCdsModel(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
		return getProvider(CdsModelProvider.class).get(userInfo, featureTogglesInfo);
	}

	@Override
	public CdsEnvironment getEnvironment() {
		return environment;
	}

	@Override
	public ParameterInfo getProvidedParameterInfo() {
		ParameterInfo parameterInfo = getProvider(ParameterInfoProvider.class).get();
		return parameterInfo != null ? parameterInfo : ParameterInfo.create();
	}

	@Override
	public UserInfo getProvidedUserInfo() {
		UserInfo userInfo = getProvider(UserInfoProvider.class).get();
		return userInfo != null ? userInfo : UserInfo.create();
	}

	@Override
	public AuthenticationInfo getProvidedAuthenticationInfo() {
		return getProvider(AuthenticationInfoProvider.class).get();
	}

	@Override
	public FeatureTogglesInfo getFeatureTogglesInfo(UserInfo userInfo, ParameterInfo parameterInfo) {
		FeatureTogglesInfo featureTogglesInfo = getProvider(FeatureTogglesInfoProvider.class).get(userInfo, parameterInfo);
		return featureTogglesInfo != null ? featureTogglesInfo : FeatureTogglesInfo.create();
	}

	@Override
	public String getLocalizedMessage(String code, Object[] args, Locale locale) {
		return getProvider(LocalizedMessageProvider.class).get(code, args, locale);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T extends CdsProvider<T>> T getProvider(Class<T> clazz) {
		return (T) providers.get(clazz);
	}

	@Override
	public RequestContextRunner requestContext() {
		return new RequestContextRunnerImpl(this);
	}

	@Override
	public ChangeSetContextRunner changeSetContext() {
		return new ChangeSetContextRunnerImpl(this);
	}

	// These methods are made available through CdsRuntimeConfigurer
	// They are therefore NOT public, but only visible in this package

	void setCdsModel(String csnPath) {
		this.cdsModel = CdsModelUtils.loadCdsModel(CdsModelUtils.buildCdsModelReaderConfig(this), csnPath);
	}

	void setCdsModel(CdsModel model) {
		this.cdsModel = model;
	}

	void registerService(Service service) {
		serviceCatalog.register(service);
		ServiceSPI serviceSPI = CdsServiceUtils.getServiceSPI(service);
		if(serviceSPI != null) {
			serviceSPI.setCdsRuntime(this);
		}
	}

	<T> void registerEventHandler(Class<T> handlerClass, Supplier<T> handlerFactory) {
		HandlerRegistryTools.registerClass(handlerClass, handlerFactory, serviceCatalog);
		if(handlerClass.getPackage().getName().startsWith(BASE_PACKAGE)) {
			logger.debug("Registered handler class {}", handlerClass.getName());
		} else {
			logger.info("Registered handler class {}", handlerClass.getName());
		}
	}

	<T extends CdsProvider<T>> void registerProvider(CdsProvider<T> provider) {
		Class<T> clazz = determineProviderClass(provider);
		T current = getProvider(clazz);
		provider.setPrevious(current);
		providers.put(clazz, provider);

		if (provider instanceof LocalizedMessageProvider) {
			ServiceExceptionUtilsImpl.defaultLocalizedMessageProvider = (LocalizedMessageProvider) provider;
		}
	}

	Map<Class<? extends CdsProvider<?>>, CdsProvider<?>> getProviders() {
		return providers;
	}

	@SuppressWarnings({"unchecked", "rawtypes"})
	private static <T extends CdsProvider<T>> Class<T> determineProviderClass(CdsProvider<T> p) {
		TypeToken<?> token = TypeToken.of(p.getClass()).getSupertype(CdsProvider.class);
		TypeToken<?> resolved = token.resolveType(CdsProvider.class.getTypeParameters()[0]);
		return (Class<T>) (Class) resolved.getRawType();
	}

	private class DefaultModelProvider implements CdsModelProvider {
		@Override
		public CdsModel get(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
			return cdsModel;
		}
	}

	private class CdsEnvironmentImpl implements CdsEnvironment {

		private final CdsProperties properties;
		private final PropertiesProvider propertiesProvider;

		public CdsEnvironmentImpl(PropertiesProvider propertiesProvider) {
			if(propertiesProvider == null) {
				this.propertiesProvider = new SimplePropertiesProvider();
			} else {
				this.propertiesProvider = propertiesProvider;
			}

			this.properties = this.propertiesProvider.bindPropertyClass("cds", CdsProperties.class);
			ServiceExceptionUtilsImpl.errorsProperties = this.properties.getErrors();
		}

		@Override
		public CdsProperties getCdsProperties() {
			return properties;
		}

		@Override
		public <T> T getProperty(String key, Class<T> asClazz, T defaultValue) {
			return propertiesProvider.getProperty(key, asClazz, defaultValue);
		}

		@Override
		public Stream<ServiceBinding> getServiceBindings() {
			return getProvider(ServiceBindingProvider.class).get();
		}

		@Override
		public ApplicationInfo getApplicationInfo() {
			return getProvider(ApplicationInfoProvider.class).get();
		}

	}

}
