package com.sap.cds.services.impl.utils;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.slf4j.helpers.MessageFormatter;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.CdsDataProcessor.Filter;
import com.sap.cds.CdsDataProcessor.Mode;
import com.sap.cds.CdsDataProcessor.Validator;
import com.sap.cds.impl.parser.PathParser;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.OnConditionAnalyzer;

/**
 * Validation utilities.
 */
public class ValidatorUtils {

	@FunctionalInterface
	public interface Handler {
		void accept(Path path, CdsElement element, String parent);
	}

	/**
	 * Returns true, if the provided {@link CdsElement} requires a not null check
	 * based on its type, potential default value or whether its draft element.
	 *
	 * @param element the cds element
	 * @return true, if element requires a not null check
	 */
	public static boolean requiresNotNullCheck(CdsElement element) {
		if (element == null) {
			return false;
		}
		// elements with default values don't have to be checked
		if (element.getType().isSimple() && element.defaultValue().isPresent()) {
			return false;
		}
		return true;
	}

	/**
	 * Runs a not null validation against the provided {@link EventContext} with the
	 * given predicate and validator implementation.
	 *
	 * @param context         the event context
	 * @param requiredByModel predicate returning whether validation is required
	 *                        based on model annotations
	 * @param validator       the validator implementation to use
	 */
	public static void runNotNullCheck(EventContext context, Predicate<CdsElement> requiredByModel,
			Validator validator) {
		CdsDataProcessor processor = CdsDataProcessor.create().addValidator(requiresNotNullCheck(requiredByModel), validator, Mode.DECLARED);
		ValidatorExecutor.processResolvedEntities(processor, context);
	}

	/**
	 * Runs the given instance of a @{@link CdsDataProcessor} against the entity that is the target of the given event context
	 * or action or function arguments wrapped in the virtual CDS type.
	 *
	 * @param processor An instance of the @{@link CdsDataProcessor} that will process the entity.
	 * @param context Instance of the @{@link EventContext} from where the entity will be extracted.
	 */
	public static void process(CdsDataProcessor processor, EventContext context) {
		ValidatorExecutor.processEntities(processor, context);
	}

	private static Filter requiresNotNullCheck(Predicate<CdsElement> requiredByModel) {
		return (path, element, type) -> {
			if (!requiredByModel.test(element)) {
				return false;
			}
			Iterator<ResolvedSegment> segments = path.reverse();
			String lastSegment = segments.next().segment().id();
			if (segments.hasNext()) {
				Optional<CdsElement> optAssociation = segments.next().type().findAssociation(lastSegment);
				if (optAssociation.isPresent()) {
					CdsElement association = optAssociation.get();
					List<CdsElement> keyElements = CdsModelUtils.getAssociationKeys(association);
					if (keyElements.contains(element) || keyElements.stream().anyMatch(k -> element.getName().equals(CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrDefault(k)))) {
						// key elements of the association target are not mandatory,
						// as they get auto-filled
						return false;
					}
				}
			}
			return true;
		};
	}

	/**
	 * Handles the given {@link CdsErrorStatuses} by either generating an error
	 * message or throwing a {@link ErrorStatusException}.
	 *
	 * @param context  the event context
	 * @param path     the relative path to the target element
	 * @param element  the target element
	 * @param statuses the cds error status
	 * @param args     message args
	 */
	public static void handleValidationError(EventContext context, Path path, CdsElement element,
			CdsErrorStatuses statuses, Object... args) {
		if (context.getCdsRuntime().getEnvironment().getCdsProperties().getErrors().isCombined()) {
			context.getMessages().error(getLocalizedMessage(context, statuses, args)).code(statuses.getCodeString())
					.target(path, element);
		} else {
			throw new ErrorStatusException(statuses, args).messageTarget(path, element);
		}
	}

	/*
	 * Returns the localized message of an {@link ErrorStatus}.
	 *
	 * @param context the event context
	 *
	 * @param errorStatus the error status
	 *
	 * @param args message args
	 *
	 * @return localized error status message
	 */
	private static String getLocalizedMessage(EventContext context, ErrorStatus errorStatus, Object... args) {
		String localized = context.getCdsRuntime().getLocalizedMessage(errorStatus.getCodeString(), args,
				context.getParameterInfo().getLocale());

		if (Objects.equals(localized, errorStatus.getCodeString())) {
			return MessageFormatter.arrayFormat(errorStatus.getDescription(), args).getMessage();
		}
		return localized;
	}

	// returns all invalid fk elements of a forward association element, filtering
	// them with the given "isNull" predicate
	//
	// TODO: revisit behavior of OnConditionAnalyzer for structured OData
	private static List<CdsElement> invalidFkElementsOfFwdAssociation(boolean isPartialUpdate, Path path,
			CdsElement element, Predicate<Object> isNull) {

		boolean isFwdAssociation = element.getType().isAssociation()
				&& !com.sap.cds.util.CdsModelUtils.isReverseAssociation(element);

		// make sure to validate only forward associations
		if (isFwdAssociation) {
			Map<String, Object> parentValues = path.target().values();

			Set<String> fkElements = new OnConditionAnalyzer(element, false).getFkMapping().keySet();

			return fkElements.stream().filter(e -> {
				if (!isPartialUpdate) {
					// insert with missing fk element, or explicitly set to null or empty string
					return !parentValues.containsKey(e) || isNull.test(parentValues.get(e));
				} else {
					// (partial) update with fk element explicitly set to null or empty string
					return parentValues.containsKey(e) && isNull.test(parentValues.get(e));
				}
			}).map(e -> com.sap.cds.util.CdsModelUtils.element(element.getDeclaringType(), PathParser.segments(e)))
					.collect(Collectors.toList());
		}
		return Collections.emptyList();
	}

	/**
	 * Returns a validator for mandatory annotated elements or elements with not
	 * null constraints. Validated elements can be of type association and requires
	 * a not null (or empty string) check on its foreign key elements.
	 *
	 * @param context      the event context
	 * @param isNull       predicate checking whether an object is set to null (or
	 *                     an empty string)
	 * @param errorHandler error handler that takes the element and
	 *                     entity name that is validated
	 * @return a validator implementation
	 */
	public static Validator assertNotNull(EventContext context, Predicate<Object> isNull,
			Handler errorHandler) {
		return (path, element, value) -> {

			// To check / possible scenarios:
			// - element explicitly set to null (or empty string)
			// - insert with missing non-association element
			// - insert with missing association element
			// - (partial) update with missing association element

			boolean isPartialUpdate = CqnService.EVENT_UPDATE.equals(context.getEvent());
			boolean isAssociation = element.getType().isAssociation();
			boolean notInRequestBody = value == CdsDataProcessor.ABSENT;

			// mandatory element explicitly set to null (or empty string), or (-> error)
			// insert with a missing non-association element                  (-> error)
			if (isNull.test(value) || !isPartialUpdate && !isAssociation && notInRequestBody) {
				errorHandler.accept(path, element, element.getDeclaringType().getQualifiedName());

			// insert with missing association element, or       (-> check fk elements)
			// (partial) update with missing association element (-> check fk elements)
			} else if (notInRequestBody) {
				for (CdsElement e : invalidFkElementsOfFwdAssociation(isPartialUpdate, path, element, isNull)) {
					errorHandler.accept(path, e, element.getDeclaringType().getQualifiedName());
				}
			}
			// (partial) update with missing non-association element (-> ignored)
		};
	}
}
