/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.draft;

import static com.sap.cds.ql.CQL.and;
import static com.sap.cds.ql.CQL.constant;
import static com.sap.cds.ql.CQL.get;
import static com.sap.cds.ql.CQL.not;
import static com.sap.cds.ql.CQL.userId;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.sap.cds.Result;
import com.sap.cds.impl.builder.model.InPredicate;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnComparisonPredicate.Operator;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.Modifier;
import com.sap.cds.ql.impl.SelectBuilder;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.draft.DraftAdministrativeData;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.model.CdsModelUtils;
import com.sap.cds.util.CqnStatementUtils;

/**
 * Used in {@link DraftActionsHandler#defaultRead(com.sap.cds.services.cds.CdsReadEventContext)} to handle reading of sanitized actives.
 */
class DraftActivesReader {

	private static final String DRAFT_ADMINISTRATIVE_DATA_CREATED_BY_USER = Drafts.DRAFT_ADMINISTRATIVE_DATA + "." + DraftAdministrativeData.CREATED_BY_USER;
	private static final String DRAFT_ADMINISTRATIVE_DATA_LAST_CHANGE_DATE_TIME = Drafts.DRAFT_ADMINISTRATIVE_DATA + "." + DraftAdministrativeData.LAST_CHANGE_DATE_TIME;

	public static DraftActivesReader create(EventContext context) {
		return new DraftActivesReader(context);
	}

	private final EventContext context;
	private final DraftInactivesReader inactivesReader;

	private DraftActivesReader(EventContext context) {
		this.context = context;
		this.inactivesReader = DraftInactivesReader.create(context);
	}

	public Result allActives(CqnSelect select, Map<String, Object> cqnNamedValues, CqnPredicate remainingWhere, CqnStructuredTypeRef remainingRef) {
		// read actives
		ActiveSanitizer sanitizer = new ActiveSanitizer(remainingWhere, remainingRef, context.getTarget(), select.excluding());
		CqnSelect sanitizedSelect = CQL.copy(select, sanitizer);
		Result result = DraftServiceImpl.downcast(context.getService()).readActive(sanitizedSelect, cqnNamedValues);

		if (result.rowCount() > 0) {
			// read corresponding drafts
			List<? extends Map<String, Object>> drafts;
			if (!sanitizer.getDraftItems().isEmpty()) {
				drafts = inactivesReader.draftsOfAllUsers(Select.from(context.getTarget()).columns(sanitizer.getDraftItems()).where(matchingKeys(result))).list();
			} else {
				drafts = Collections.emptyList();
			}

			// restore draft fields on active result
			traverse(result.list(), drafts, context.getTarget(), sanitizer.getSelectionTree());
		}
		return result;
	}

	public Result withoutDrafts(CqnSelect select, Map<String, Object> cqnNamedValues, CqnPredicate remainingWhere, CqnStructuredTypeRef remainingRef) {
		// read all existing drafts in edit scenario
		Result drafts = inactivesReader.draftsOfAllUsers(Select.from(remainingRef)
			.columns(keys(context.getTarget()).toArray(new String[0]))
			.where(get(Drafts.HAS_ACTIVE_ENTITY).eq(true).and(remainingWhere))
			.search(select.search().orElse(null)));

		// read actives, not having a draft
		ActiveSanitizer sanitizer = new ActiveSanitizer(remainingWhere, remainingRef, context.getTarget(), select.excluding());
		CqnSelect sanitizedSelect = SelectBuilder.copy(select, sanitizer).filter(not(matchingKeys(drafts)));
		Result result = DraftServiceImpl.downcast(context.getService()).readActive(sanitizedSelect, cqnNamedValues);

		// restore draft fields on active result
		traverse(result.list(), Collections.emptyList(), context.getTarget(), sanitizer.getSelectionTree());
		return result;
	}

	public Result withDraftsFromOtherUsers(CqnSelect select, Map<String, Object> cqnNamedValues, CqnPredicate remainingWhere, CqnStructuredTypeRef remainingRef, boolean locked) {
		ActiveSanitizer sanitizer = new ActiveSanitizer(remainingWhere, remainingRef, context.getTarget(), select.excluding());
		SelectBuilder<?> sanitizedSelect = SelectBuilder.copy(select, sanitizer);

		// read all existing drafts..
		Result drafts = inactivesReader.draftsOfAllUsers(Select.from(remainingRef)
			.columns(addIfMissing(sanitizer.getDraftItems(), keys(context.getTarget()))) // ensure keys
			.where(and(
				remainingWhere,
				// ..in edit scenario
				get(Drafts.HAS_ACTIVE_ENTITY).eq(true).and(
				// ..owned by another user
				ownedbyAnotherUser(locked, context))))
			.search(select.search().orElse(null))
			.orderBy(select.orderBy())
			.limit(select.top(), select.skip()));

		if (drafts.rowCount() > 0) {
			// read actives for corresponding drafts
			// we rely on pagination through the key-based where condition
			CqnSelect selectActivesFromDrafts = sanitizedSelect.filter(matchingKeys(drafts)).limit(-1);
			Result result = DraftServiceImpl.downcast(context.getService()).readActive(selectActivesFromDrafts, cqnNamedValues);

			// restore draft fields on active result
			traverse(result.list(), drafts.list(), context.getTarget(), sanitizer.getSelectionTree());
			return result;
		} else {
			return DraftHandlerUtils.buildNoOpResult(context);
		}
	}

	public static CqnPredicate ownedbyAnotherUser(boolean locked, EventContext context) {
		return CQL.and(List.of(
			// ..not owned by current user
			get(DRAFT_ADMINISTRATIVE_DATA_CREATED_BY_USER).isNotNull(),
			get(DRAFT_ADMINISTRATIVE_DATA_CREATED_BY_USER).ne(userId()),
			// ..locked (last change before timeout) or not locked (last change after timeout)
			CQL.comparison(get(DRAFT_ADMINISTRATIVE_DATA_LAST_CHANGE_DATE_TIME), locked ? Operator.GT : Operator.LE, constant(DraftModifier.getCancellationThreshold(context)))
		));
	}

	CqnPredicate matchingKeys(Result result) {
		List<String> keyElements = keys(context.getTarget());

		return InPredicate.in(keyElements, result.list());
	}

	private static void traverse(List<? extends Map<String, Object>> rows, List<? extends Map<String, Object>> drafts, CdsEntity entity, SelectionNode node) {
		List<String> keyElements = keys(entity);
		for (Map<String, Object> row : rows) {
			Map<String, Object> matchingDraft = null;
			if (!drafts.isEmpty()) {
				Map<String, Object> keys = new HashMap<>(keyElements.size());
				keyElements.forEach(key -> keys.put(key, row.get(key)));
				matchingDraft = findMatching(drafts, keys);
			}

			if (node.draftElements().contains(Drafts.IS_ACTIVE_ENTITY))
				row.put(Drafts.IS_ACTIVE_ENTITY, true);
			if (node.draftElements().contains(Drafts.HAS_ACTIVE_ENTITY))
				row.put(Drafts.HAS_ACTIVE_ENTITY, false);
			if (node.draftElements().contains(Drafts.HAS_DRAFT_ENTITY))
				row.put(Drafts.HAS_DRAFT_ENTITY, matchingDraft != null);
			if (node.draftElements().contains(Drafts.DRAFT_ADMINISTRATIVE_DATA))
				row.put(Drafts.DRAFT_ADMINISTRATIVE_DATA, matchingDraft != null ? matchingDraft.get(Drafts.DRAFT_ADMINISTRATIVE_DATA) : null);
			if (node.draftElements().contains(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID))
				row.put(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID, matchingDraft != null ? matchingDraft.get(Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID) : null);

			for (SelectionNode childNode : node.children()) {
				List<? extends Map<String, Object>> childRows = toList(row.get(childNode.elementName()));
				if (!childRows.isEmpty()) {
					List<? extends Map<String, Object>> childDraftRows = matchingDraft != null ? toList(matchingDraft.get(childNode.elementName())) : Collections.emptyList();
					traverse(childRows, childDraftRows, entity.getTargetOf(childNode.elementName()), childNode);
				}
			}
		}
	}

	private static Map<String, Object> findMatching(List<? extends Map<String, Object>> drafts, Map<String, Object> keys) {
		Map<String, Object> matchingDraft = null;
		for (Map<String, Object> draft : drafts) {
			if (draft.entrySet().containsAll(keys.entrySet())) {
				matchingDraft = draft;
				break;
			}
		}
		return matchingDraft;
	}

	static List<String> keys(CdsEntity entity) {
		return com.sap.cds.util.CdsModelUtils.keyNames(entity).stream().filter(k -> !k.equals(Drafts.IS_ACTIVE_ENTITY)).toList();
	}

	private static List<CqnSelectListItem> addIfMissing(List<CqnSelectListItem> items, List<String> itemsToAdd) {
		Set<String> contained = items.stream().flatMap(CqnSelectListItem::ofRef).map(CqnElementRef::path).collect(Collectors.toSet());
		itemsToAdd.stream().filter(i -> !contained.contains(i)).map(CQL::get).forEach(items::add);
		return items;
	}

	@SuppressWarnings("unchecked")
	private static List<? extends Map<String, Object>> toList(Object value) {
		if (value instanceof List) {
			return (List<? extends Map<String, Object>>) value;
		} else if (value instanceof Map) {
			return Collections.singletonList((Map<String, Object>) value);
		} else {
			return Collections.emptyList();
		}
	}

	private record SelectionNode(String elementName, Set<String> draftElements, List<SelectionNode> children) {

		public SelectionNode(String elementName) {
			this(elementName, new HashSet<>(), new ArrayList<>());
		}

	}

	private static class ActiveSanitizer implements Modifier {

		private record Items(List<CqnSelectListItem> items, List<CqnSelectListItem> draftItems, boolean keysRequired) {}

		private final CqnPredicate remainingWhere;
		private final CqnStructuredTypeRef remainingRef;
		private final CdsEntity target;
		private final List<String> targetExcluding;
		private final SelectionNode selectionTree = new SelectionNode(null);
		private List<CqnSelectListItem> draftItems;

		public ActiveSanitizer(CqnPredicate remainingWhere, CqnStructuredTypeRef remainingRef, CdsEntity target, List<String> targetExcluding) {
			this.remainingWhere = remainingWhere;
			this.remainingRef = remainingRef;
			this.target = target;
			this.targetExcluding = targetExcluding;
		}

		public SelectionNode getSelectionTree() {
			return selectionTree;
		}

		public List<CqnSelectListItem> getDraftItems() {
			return draftItems;
		}

		@Override
		public CqnStructuredTypeRef ref(CqnStructuredTypeRef ref) {
			return remainingRef;
		}

		@Override
		public List<CqnSelectListItem> items(List<CqnSelectListItem> items) {
			var pair = splitItems(items, target, targetExcluding, selectionTree);
			draftItems = pair.draftItems();
			return pair.items();
		}

		private Items splitItems(List<CqnSelectListItem> items, CdsEntity entity, List<String> excluding, SelectionNode node) {
			// resolve star to easily track draft elements
			items = CqnStatementUtils.resolveStar(items, excluding, entity, false);
			// split SLIs into draft elements and non draft elements
			List<CqnSelectListItem> nonDraft = new ArrayList<>(items.size());
			List<CqnSelectListItem> draft = new ArrayList<>(3);
			boolean keysRequired = false;
			for (CqnSelectListItem item : items) {
				if (item.isRef()) {
					String path = item.asRef().path();
					if (Drafts.ELEMENTS.contains(path)) {
						node.draftElements().add(path);
						if (Drafts.DRAFT_ADMINISTRATIVE_DATA.equals(path) ||
								Drafts.DRAFT_ADMINISTRATIVE_DATA_DRAFT_UUID.equals(path)) {
							draft.add(item);
							keysRequired = true;
						} else if (Drafts.HAS_DRAFT_ENTITY.equals(path)) {
							keysRequired = true;
						}
					} else {
						nonDraft.add(item);
					}
				} else if (item.isExpand()) {
					CqnExpand expand = item.asExpand();
					String path = expand.ref().path();
					if (Drafts.DRAFT_ADMINISTRATIVE_DATA.equals(path)) {
						node.draftElements().add(Drafts.DRAFT_ADMINISTRATIVE_DATA);
						draft.add(expand);
						keysRequired = true;
					} else {
						SelectionNode expandNode = new SelectionNode(path);
						// split expand into draft elements and non draft elements
						var pair = splitItems(expand.items(), CdsModelUtils.getRefTarget(expand.ref(), entity), Collections.emptyList(), expandNode);
						List<CqnSortSpecification> sortSpecs = orderBy(expand.orderBy());
						if (!pair.items().isEmpty()) {
							nonDraft.add(CQL.copy(expand).items(pair.items()).orderBy(sortSpecs));
						}
						if (!pair.draftItems().isEmpty()) {
							draft.add(CQL.copy(expand).items(pair.draftItems()).orderBy(sortSpecs));
						}
						if (!expandNode.children().isEmpty() || !expandNode.draftElements().isEmpty()) {
							node.children().add(expandNode);
						}
						if (pair.keysRequired()) {
							keysRequired = true;
						}
					}
				} else {
					nonDraft.add(item);
				}
			}

			// ensure keys are selected
			if (keysRequired) {
				List<String> keys = keys(entity);
				addIfMissing(nonDraft, keys);
				keys.stream().map(CQL::get).forEach(draft::add);
			}
			return new Items(nonDraft, draft, keysRequired);
		}

		@Override
		public CqnPredicate where(Predicate where) {
			return remainingWhere;
		}

		@Override
		public List<CqnSortSpecification> orderBy(List<CqnSortSpecification> sortSpecs) {
			return sortSpecs.stream()
				.filter(s -> !s.value().isRef() || !s.value().asRef().path().equals(Drafts.IS_ACTIVE_ENTITY))
				.toList();
		}

	}

}
