package com.sap.cds.services.impl.outbox;

import static com.sap.cds.services.utils.CdsErrorStatuses.OUTBOX_SERVICE_NOT_OUTBOXABLE;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.Set;

import com.sap.cds.services.Service;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.ErrorStatusException;

public class OutboxedServiceProxyUtils {

	private OutboxedServiceProxyUtils() {
		// prevent instantiation
	}

	@SuppressWarnings("unchecked")
	public static <S extends Service> S unboxed(S service) {
		if (Proxy.isProxyClass(service.getClass())) {
			InvocationHandler invocationHandler = Proxy.getInvocationHandler(service);
			if (invocationHandler instanceof OutboxedServiceInvocationHandler serviceInvocationHandler) {
				return (S) serviceInvocationHandler.getDelegatedService();
			}
		}
		return service;
	}

	@SuppressWarnings("unchecked")
	public static <S extends Service> S outboxed(OutboxService outboxService, S service, CdsRuntime runtime) {
		ensureNotOutboxService(service);

		S serviceForOutboxing = service;
		if (Proxy.isProxyClass(service.getClass())) {
			InvocationHandler invocationHandler = Proxy.getInvocationHandler(service);
			if (invocationHandler instanceof OutboxedServiceInvocationHandler serviceInvocationHandler) {
				if (serviceInvocationHandler.getOutboxService() == outboxService) {
					return service; // already outboxed with correct outbox
				}
				serviceForOutboxing = (S) serviceInvocationHandler.getDelegatedService();
			}
		}

		OutboxedServiceInvocationHandler invocationHandler = new OutboxedServiceInvocationHandler(outboxService, serviceForOutboxing, runtime);
		return (S) Proxy.newProxyInstance(service.getClass().getClassLoader(), getInterfaces(service), invocationHandler);
	}

	private static <S extends Service> void ensureNotOutboxService(S service) {
		if (service instanceof OutboxService) {
			throw new ErrorStatusException(OUTBOX_SERVICE_NOT_OUTBOXABLE, service.getName());
		}
	}

	private static Class<?>[] getInterfaces(Service service) {
		Set<Class<?>> interfaces = new LinkedHashSet<>();
		Class<?> clazz = service.getClass();

		do {
			Set<Class<?>> nextInterfaces = new LinkedHashSet<>(Arrays.asList(clazz.getInterfaces()));
			if (interfaces.size() == 1) {
				boolean allAssignable = true;
				Class<?> current = interfaces.iterator().next();
				for (Class<?> nextInterface : nextInterfaces) {
					if (!nextInterface.isAssignableFrom(current)) {
						allAssignable = false;
						break;
					}
				}
				if (!allAssignable) {
					interfaces.addAll(nextInterfaces);
				}
			} else {
				interfaces.addAll(nextInterfaces);
			}
			clazz = clazz.getSuperclass();
		} while (clazz != null);

		return interfaces.toArray(new Class<?>[0]);
	}
}
