/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import static com.sap.cds.reflect.CdsBaseType.UUID;
import static org.apache.commons.lang3.StringUtils.unwrap;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVFormat.Builder;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.io.input.BOMInputStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Upsert;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.environment.CdsProperties.DataSource.Csv;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CdsTypeUtils;
import com.sap.cds.util.DataUtils;
public class CsvDataLoader {

	private final static Logger logger = LoggerFactory.getLogger(CsvDataLoader.class);

	private final PersistenceService db;
	private final CdsRuntime runtime;
	private final Csv config;
	private final char[] supportedDelimiters = { ';', ',', '\t'};

	public CsvDataLoader(PersistenceService db, CdsRuntime runtime) {
		this.db = db;
		this.runtime = runtime;
		this.config = runtime.getEnvironment().getCdsProperties().getDataSource().getCsv();
	}

	/**
	 * Loads CSVs from the first existing CSV folder location.
	 * The paths is resolved based on the current working directory
	 */
	public void load() {
		if (config.isSingleChangeset()) {
			// Run the whole CSV data import in a single changeset to avoid possible foreign key constraint violations,
			// if assert_integrity in DB is enabled
			runtime.changeSetContext().run(t -> {
				load(t);
			});
		} else {
			// import each CSV file in it's own changeset
			load(null);
		}
	}

	private void load(ChangeSetContext ctx) {
		for(String path : config.getPaths()) {
			boolean recursive = false;
			if(path.endsWith("**") && path.length() > 2) {
				path = path.substring(0, path.length() - 2);
				recursive = true;
			}

			File folder = new File(path);
			if(folder.exists() && folder.isDirectory()) {
				loadFolder(folder, recursive, ctx);
			}
		}
	}

	/**
	 * Loads CSV files from within the specified folder
	 * @param folder the folder
	 * @param recursive true, if the folder should be scanned recursively
	 * @param ctx an optional {@link ChangeSetContext}
	 */
	public void loadFolder(File folder, boolean recursive, ChangeSetContext ctx) {
		String fileSuffix = config.getFileSuffix();
		for (File file : folder.listFiles()) {
			if (file.isFile() && file.getName().endsWith(fileSuffix)) {
				loadFile(file, ctx);
			}

			if (recursive && file.isDirectory()) {
				loadFolder(file, true, ctx);
			}
		}
	}

	/**
	 * Loads CSV files from the specified file (extension needs to be .csv)
	 * @param file the file
	 * @param ctx an optional {@link ChangeSetContext}
	 */
	void loadFile(File file, ChangeSetContext ctx) {
		if (file.length() == 0) {
			return;
		}
		try (CSV csv = new CSV(file)) {
			List<Map<String, Object>> entries = csv.data().collect(Collectors.toList());
			if (!entries.isEmpty()) {
				if (ctx == null) {
					try {
						runtime.changeSetContext().run(context -> {
							loadFile(csv, entries, context);
						});
					} catch (ServiceException e) {
						logger.debug("Skipped filling {} from {}", csv.entity.getQualifiedName(), file.getPath(), e);
					}
				} else {
					loadFile(csv, entries, ctx);
				}
			}
		} catch (UncheckedIOException | IOException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE, file.getPath(), e); // NOSONAR
		}
	}

	private void loadFile(CSV csv, List<Map<String, Object>> entries, ChangeSetContext ctx) {
		if ("always".equals(config.getInitializationMode())) {
			Upsert upsert = Upsert.into(csv.entity()).entries(entries).hint("cross-tenant", true);
			db.run(upsert);
		} else {
			Insert insert = Insert.into(csv.entity()).entries(entries).hint("cross-tenant", true);
			db.run(insert);
		}

		if(ctx.isMarkedForCancel()) {
			logger.debug("Cancelled filling {} from {}", csv.entity().getQualifiedName(), csv.file.getPath());
		} else {
			logger.info("Filling {} from {}", csv.entity().getQualifiedName(), csv.file.getPath());
		}
	}

	private class CSV implements AutoCloseable {

		private final CdsEntity entity;
		private final File file;
		private final BufferedReader br;
		private final List<ElementPath> headers;
		private CSVParser csvParser;
		private final Builder formatTemplate = Builder.create(CSVFormat.RFC4180)
			.setEscape('\\')
			.setCommentMarker('#')
			.setIgnoreEmptyLines(true)
			.setAllowMissingColumnNames(true);

		@SuppressWarnings({ "deprecation", "resource" })
		public CSV(File file) throws IOException {
			this.file = file;
			br = new BufferedReader(
					new InputStreamReader(new BOMInputStream(Files.newInputStream(file.toPath())), StandardCharsets.UTF_8));

			String entityName = entityName(file.getName());
			entity = runtime.getCdsModel().findEntity(entityName).orElseGet(() -> {
				String theEntityName = entityName;
				if (theEntityName.lastIndexOf(".texts") > 0) {
					Optional<CdsEntity> entity = runtime.getCdsModel().findEntity(
							theEntityName.substring(0, theEntityName.lastIndexOf(".texts") + 6));
					if (entity.isPresent()) {
						return entity.get();
					}
				}
				if (theEntityName.lastIndexOf("_texts") > 0) {
					theEntityName = theEntityName.substring(0, theEntityName.lastIndexOf("_texts") + 6);
					Optional<CdsEntity> entity = runtime.getCdsModel().findEntity(theEntityName);
					if (entity.isPresent()) {
						return entity.get();
					}
				}
				if (theEntityName.endsWith("_texts")) {
					// replace entity suffix "_texts" with ".texts"
					String altEntityName = theEntityName.substring(0, theEntityName.length() - "_texts".length()) + ".texts";
					return runtime.getCdsModel().findEntity(altEntityName).orElse(null);
				}
				return null;
			});
			if (entity == null) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_ENTITYNOTFOUND, entityName, file.getPath()); // NOSONAR
			}

			headers = parseHeaders();
		}

		public CdsEntity entity() {
			return entity;
		}

		public Stream<Map<String, Object>> data() {
			if(headers.isEmpty()) {
				return Stream.empty();
			}
			try {
				return csvParser.getRecords().stream().map(this::convert);
			} catch (UncheckedIOException e) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE, file.getPath(), e);
			}
		}

		private List<ElementPath> parseHeaders() throws IOException {
			// Common CSV can detect headers only when commas used as the delimiter
			String headerLine;

			while ((headerLine = br.readLine()) != null && headerLine.isEmpty()) {}

			if (headerLine == null) {
				return Collections.emptyList();
			} else {
				String delimiter = detectDelimiter(headerLine);
				csvParser = new CSVParser(br,
					formatTemplate
						.setHeader(splitHeaders(headerLine, delimiter))
						.setDelimiter(delimiter)
						.build());

				return csvParser.getHeaderNames().stream()
					.filter(StringUtils::isNotBlank)
					.map(n -> path(entity, n))
					.collect(Collectors.toList());
			}
		}

		private String detectDelimiter(String line) {
			char result = ';';
			int delimitersCount = 0;
			for(char current : supportedDelimiters) {
				if(line.indexOf(current) >= 0) {
					result = current;
					++delimitersCount;
				}
			}
			if (delimitersCount > 1) {
				throw new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_INVALIDHEADER, file.getPath());
			}
			return String.valueOf(result);
		}

		private Map<String, Object> convert(CSVRecord csvRecord) {
			Map<String, Object> row = new HashMap<>();
			for (ElementPath header : headers) {
				if (csvRecord.isSet(header.csvName)) {
					String value = csvRecord.get(header.csvName);
					if (StringUtils.isNotEmpty(value)) {
						handleValue(row, header.element, header.path, value);
					}
				}
			}
			return row;
		}

		private void handleValue(Map<String, Object> row, CdsElement element, String path, String value) {
			CdsType type = element.getType();
			if (type.isSimpleType(UUID) && !CdsTypeUtils.isStrictUUID(element, type)) {
				direct(row, path, type.as(CdsSimpleType.class), value);
			} else if (type.isSimple()) {
				simple(row, path, type.as(CdsSimpleType.class), value);
			} else if (type.isStructured()) {
				structured(row, path, type.as(CdsStructuredType.class), value);
			} else if (type.isArrayed()) {
				arrayed(row, path, element, value);
			} else if (type.isAssociation()) {
				association(row, path, element, value);
			}
		}

		private void direct(Map<String, Object> row, String path, CdsSimpleType type, String value) {
			try {
				DataUtils.resolvePathAndAdd(row, path, value);
			} catch (Exception e) {
				throw error(path, type, false, value, e);
			}
		}

		private void simple(Map<String, Object> row, String path, CdsSimpleType type, String value) {
			try {
				Object val = CdsTypeUtils.parse(type.getType(), value);
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, type, false, value, e);
			}
		}

		private void structured(Map<String, Object> row, String path, CdsStructuredType type, String value) {
			try {
				Map<String, Object> val = StructDataParser.create(type).parseObject(value);
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, type, false, value, e);
			}
		}

		private void arrayed(Map<String, Object> row, String path, CdsElement element, String value) {
			CdsType itemsType = element.getType().as(CdsArrayedType.class).getItemsType();
			try {
				Object val = StructDataParser.parseArrayOf(itemsType, value);
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, itemsType, true, value, e);
			}
		}

		private void association(Map<String, Object> row, String path, CdsElement element, String value) {
			CdsEntity targetType = element.getType().as(CdsAssociationType.class).getTarget();
			StructDataParser parser = StructDataParser.create(targetType);
			boolean isSingleValued = CdsModelUtils.isSingleValued(element.getType());
			try {
				Object val = isSingleValued ? parser.parseObject(value) : parser.parseArray(value);
				DataUtils.resolvePathAndAdd(row, path, val);
			} catch (Exception e) {
				throw error(path, targetType, !isSingleValued, value, e);
			}
		}

		private ErrorStatusException error(String element, CdsType type, boolean array, String value, Exception e) {
			return new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_TYPEMISMATCH, value,
					array ? type.getName() + "[]" : type.getName(), element, file.getPath(), e); // NOSONAR
		}

		private String entityName(String fileName) {
			return fileName.replace("-", ".").substring(0, fileName.length() - 4);
		}

		private ElementPath path(CdsStructuredType entity, String column) {
			// Headers in CSV file are not normalized e.g. may have different case or surrounding spaces.
			// The element name is trimmed, but ElementPath keeps the original column name so the CSV lib
			// can later find the headers in instances of CSVRecord
			String elementName = column.trim();
			Optional<ElementPath> path = findElement(entity, elementName).map(p -> new ElementPath(entity.getElement(p), p, column));
			return path.orElseThrow(() -> new ErrorStatusException(CdsErrorStatuses.INVALID_CSV_FILE_UNKNOWNCOLUMN,
					file.getPath(), elementName, entity.getQualifiedName()));
		}

		private Optional<String> findElement(CdsStructuredType entity, String column) {
			column = column.toUpperCase(Locale.US);
			Map<String, CdsElement> normalized = normalize(entity);

			CdsElement element = normalized.get(column);
			if (element != null) {
				return Optional.of(element.getName());
			}

			for (Map.Entry<String, CdsElement> entry : normalized.entrySet()) {
				String n = entry.getKey();
				CdsElement e = entry.getValue();
				if (column.startsWith(n + "_")) {
					String suffix = column.substring(n.length() + 1);
					CdsType type = e.getType();
					Optional<String> subElement = Optional.empty();
					if (type.isStructured()) {
						subElement = findElement(type.as(CdsStructuredType.class), suffix);
					} else if (CdsModelUtils.managedToOne(type)) {
						subElement = findElement(type.as(CdsAssociationType.class).getTarget(), suffix);
					}
					if (subElement.isPresent()) {
						String path = e.getName() + "." + subElement.get();
						return Optional.of(path);
					}
				}
			}

			return Optional.empty();
		}

		private Map<String, CdsElement> normalize(CdsStructuredType type) {
			return type.elements().collect(Collectors.toMap(e -> e.getName().toUpperCase(Locale.US), e -> e));
		}

		@Override
		public void close() throws IOException {
			if (csvParser != null) {
				csvParser.close();
			}
			if (br != null)
				br.close();
		}

		private static String[] splitHeaders(String headerLine, String delimiter) {
			return Stream.of(headerLine.split(delimiter))
				.map(v -> unwrap(v, CSVFormat.RFC4180.getQuoteCharacter()))
				.toArray(String[]::new);
		}
	}

	private static class ElementPath {
		final CdsElement element;
		final String path;

		// Original column name from CSV file
		final String csvName;

		public ElementPath(CdsElement element, String path, String csvName) {
			this.element = element;
			this.path = path;
			this.csvName = csvName;
		}
	}
}
