/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import com.sap.cds.CdsDataProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;

@ServiceName(value = "*", type = PersistenceService.class)
public class FieldLengthAssertionHandler implements EventHandler {

	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		CdsDataProcessor.create()
				.addValidator(FieldLengthAssertionHandler::isStringWithLength,
						FieldLengthAssertionHandler::validateFieldLength)
				.process(CdsServiceUtils.getEntities(context), context.getTarget());
	}

	private static void validateFieldLength(Path path, CdsElement element, Object value) {
		if (value instanceof String) {
			Integer maxLength = simpleType(element).get("length");
			int valueLength = ((String) value).length();
			// check if length of given string exceeds maximum length of field
			if (valueLength > maxLength) {
				throw new ErrorStatusException(CdsErrorStatuses.EXCEEDED_FIELD_LENGTH, element.getName(), maxLength)
						.messageTarget(path, element);
			}
		}
	}

	private static boolean isStringWithLength(Path path, CdsElement element, CdsType t) {
		return t.isSimpleType(CdsBaseType.STRING) && t.as(CdsSimpleType.class).get("length") != null;
	}

	private static CdsSimpleType simpleType(CdsElement element) {
		if (element.getType().isArrayed()) {
			return element.getType().as(CdsArrayedType.class).getItemsType().as(CdsSimpleType.class);
		}
		return element.getType().as(CdsSimpleType.class);
	}

}
