/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.auditlog;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.stream.StreamSupport;

import com.sap.cds.CdsData;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.impl.diff.CdsDiffProcessor;
import com.sap.cds.impl.diff.DiffProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.auditlog.Action;
import com.sap.cds.services.auditlog.ChangedAttribute;
import com.sap.cds.services.auditlog.DataModification;
import com.sap.cds.services.runtime.CdsRuntime;

class PersonalDataAnalyzerModification extends PersonalDataAnalyzer {
	private List<? extends Map<String, Object>> oldData;

	PersonalDataAnalyzerModification(CdsStructuredType entity, PersonalDataCaches caches, CdsRuntime runtime) {
		super(entity, caches, runtime);
	}

	PersonalDataAnalyzerModification(CdsEntity entity, List<? extends Map<String, Object>> data, PersonalDataCaches caches,
		CdsRuntime runtime) {
		this(entity, caches, runtime);
		setData(data);
	}

	/**
	 * Sets the {@link List} with old or deleted data for current entity.
	 *
	 * @param oldData a {@link List} with old or deleted data
	 */
	public void setOldData(List<? extends CdsData> oldData) {
		this.oldData = oldData;
	}

	/**
	 * @return a {@link List} with modifications to personal data fields.
	 */
	public List<DataModification> getDataModifications() {
		if (Boolean.TRUE.equals(personalData.getLogInsert().isEnabled())
			|| Boolean.TRUE.equals(personalData.getLogUpdate().isEnabled())
			|| Boolean.TRUE.equals(personalData.getLogDelete().isEnabled())) {

			DataToModifications visitor = new DataToModifications(this.caches);
			DiffProcessor.create().forDeepTraversal()
				.add((path, element, type) -> {
					PersonalDataMeta meta = visitor.metadataProvider.getMeta(path.target().entity());
					if (element != null && !element.getType().isAssociation()) {
						return meta.getPersonalDataNames().contains(element.getName());
					} else {
						return meta.hasPersonalData();
					}
				}, visitor)
				.process(super.data, oldData, this.entity);
			return visitor.getResult();
		} else {
			return Collections.emptyList();
		}
	}

	/**
	 * Indicates whether the given {@link CdsEntity entity} contains personal data or not.
	 *
	 * @return <code>true</code> if given {@link CdsEntity entity} contains personal data.
	 */
	@Override
	public boolean hasPersonalData() {
		if (super.data != null) {
			return hasPersonalData(super.entity, super.data);
		}

		return getMeta(super.entity).hasPersonalData();
	}

	/**
	 * Indicates whether the given data contains personal data or not.
	 *
	 * @param entity the {@link CdsStructuredType entity} whose personal data fields will be determined
	 * @param data   the data that be checked for personal data
	 * @return <code>true</code> if personal data is found
	 */
	private boolean hasPersonalData(CdsStructuredType entity, Iterable<? extends Map<String, Object>> data) {
		AtomicBoolean hasPersonalData = new AtomicBoolean(false);

		DataProcessor.create().action(new DataProcessor.Action() {
			@Override
			public void entries(Path path, CdsElement element, CdsStructuredType type, Iterable<Map<String, Object>> dataList) {
				PersonalDataMeta meta = getMeta(type);

				if (meta != null) {
					Set<String> personal = meta.getPersonalDataNames();

					// check if we found already personal data in the provided data tree
					if (!hasPersonalData.get() && meta.hasPersonalData() && StreamSupport.stream(dataList.spliterator(), false)
						.anyMatch(dataRow -> dataRow.keySet().stream().anyMatch(personal::contains))) {
						hasPersonalData.set(true);
					}
				}
			}
		}).process(data, entity);

		return hasPersonalData.get();
	}

	private class DataToModifications implements CdsDiffProcessor.DiffVisitor {

		private final PersonalDataCaches metadataProvider;
		private final Map<Path, DataModification> result = new HashMap<>();

		DataToModifications(PersonalDataCaches metadataProvider) {
			this.metadataProvider = metadataProvider;
		}

		@Override
		public void changed(Path path, CdsElement element, Object newValue, Object oldValue) {
			if (Boolean.TRUE.equals(personalData.getLogUpdate().isEnabled())) {
				DataModification modification = result.computeIfAbsent(path, p ->
					newModification(Action.UPDATE, p.target().values(), metadataProvider.getMeta(p.target().entity())));

				ChangedAttribute attribute = ChangedAttribute.create();
				attribute.setName(element.getName());

				// nulls in the payload are not needed, audit log treats absence of a value
				// in the payload as marker that something was added or removed
				if (newValue != null) {
					attribute.setNewValue(newValue.toString());
				}
				if (oldValue != null) {
					attribute.setOldValue(oldValue.toString());
				}
				modification.getAttributes().add(attribute);
			}
		}

		@Override
		public void added(Path path, CdsElement association, Map<String, Object> newValue) {
			if (Boolean.TRUE.equals(personalData.getLogInsert().isEnabled())) {
				PersonalDataMeta meta = metadataProvider.getMeta(path.target().entity());
				DataModification modification =
					newModification(Action.CREATE, newValue, meta);
				withAttributes(modification, newValue, meta, ChangedAttribute::setNewValue);
				result.put(path, modification);
			}
		}

		@Override
		public void removed(Path path, CdsElement association, Map<String, Object> oldValue) {
			if (Boolean.TRUE.equals(personalData.getLogDelete().isEnabled())) {
				PersonalDataMeta meta = metadataProvider.getMeta(path.target().entity());
				DataModification modification =
					newModification(Action.DELETE, oldValue, meta);
				withAttributes(modification, oldValue, meta, ChangedAttribute::setOldValue);
				result.put(path, modification);
			}
		}

		public List<DataModification> getResult() {
			return new LinkedList<>(result.values());
		}

		private void withAttributes(DataModification modification,
			Map<String, Object> data, PersonalDataMeta meta, BiConsumer<ChangedAttribute, String> action) {

			modification.setAttributes(
				meta.getPersonalDataNames().stream()
					.filter(e -> data.get(e) != null)
					.map(e -> {
						ChangedAttribute attribute = ChangedAttribute.create();
						attribute.setName(e);
						action.accept(attribute, data.get(e).toString());
						return attribute;
					}).toList());
		}

		private DataModification newModification(Action action, Map<String, Object> data, PersonalDataMeta meta) {
			DataModification modification = DataModification.create();
			modification.setAction(action);
			modification.setDataObject(createDataObject(data, meta));
			modification.setDataSubject(createDataSubject(data, meta));
			modification.setAttributes(new LinkedList<>());
			return modification;
		}
	}
}
