/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.Map;
import java.util.function.Predicate;

import com.sap.cds.reflect.CdsDefinition;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.impl.utils.ValidatorUtils;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.model.CdsAnnotations;

/**
 * Handler class to check mandatory elements Mandatory elements are expected to
 * be sent by the client If mandatory elements are string based, empty strings
 * are considered invalid as well
 */
@ServiceName(value = "*", type = ApplicationService.class)
public class MandatoryHandler implements EventHandler {

	protected static Predicate<CdsElement> requiresMandatoryCheck() {
		return element -> {
			if (!ValidatorUtils.requiresNotNullCheck(element)) {
				return false;
			}
			if (CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrValue(element, null) != null) {
				return false;
			}
			boolean mandatory = CdsAnnotations.MANDATORY.isTrue(element);
			if (!mandatory) {
				Map<?, ?> commonFieldControl = CdsAnnotations.COMMON_FIELDCONTROL.getOrDefault(element);
				if ((commonFieldControl == null || !"Mandatory".equals(commonFieldControl.get("#")))) {
					return false;
				}
			}
			return true;
		};
	}

	@Before
	@HandlerOrder(OrderConstants.Before.VALIDATE_FIELDS)
	public void runCheck(EventContext context) {
		ValidatorUtils.runNotNullCheck(context, requiresMandatoryCheck(),
				ValidatorUtils.assertNotNull(context,
						value -> value == null || value instanceof String && StringUtils.isEmpty((String) value),
						(path, element, entity) -> {
					if (CdsModelUtils.isInternalOperationType(element.getDeclaringType().as(CdsDefinition.class))) {
						ValidatorUtils.handleValidationError(context, path, element,
							CdsErrorStatuses.MISSING_VALUE, element.getName());
					} else {
						ValidatorUtils.handleValidationError(context, path, element,
							CdsErrorStatuses.VALUE_REQUIRED, element.getName(), entity);
					}}));
	}
}
