/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.cds;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.sap.cds.impl.DataProcessor;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;

@ServiceName(value = "*", type = ApplicationService.class)
public class ReadOnlyHandler implements EventHandler {

	@Before(event = { CqnService.EVENT_CREATE, CqnService.EVENT_UPDATE, CqnService.EVENT_UPSERT })
	@HandlerOrder(OrderConstants.Before.FILTER_FIELDS)
	public void cleanReadOnlyFields(EventContext context) {
		String event = context.getEvent();

		List<Map<String, Object>> entries = CdsServiceUtils.getEntities(context);

		DataProcessor.create().bulkAction((struct, data) -> {
			// find all read only elements
			for (Map<String, Object> map : data) {
				Set<String> readOnlyElementNames = struct.elements().filter(element -> isReadOnly(element, event, map))
						.map(e -> e.getName()).collect(Collectors.toSet());
				if (readOnlyElementNames.isEmpty()) {
					return;
				}

				// remove read only elements from the map
				for (String e : readOnlyElementNames) {
					map.remove(e);
				}
			}
		}).process(entries, context.getTarget());
	}

	/**
	 * Returns true, if a {@link CdsElement} is read only
	 * @param element the {@link CdsElement}
	 * @param event the event currently being processed
	 * @param map the data map
	 * @return true, if a {@link CdsElement} is read only
	 */
	public boolean isReadOnly(CdsElement element, String event, Map<String, Object> map) {
		if (element == null) {
			return false;
		}

		boolean onUpdate = CdsAnnotations.ON_UPDATE.getOrDefault(element) != null;
		boolean onInsert = CdsAnnotations.ON_INSERT.getOrDefault(element) != null;

		// TODO cds.on.insert should actually be readonly during UPSERTs -> however this would remove createdAt timestamps for example during OData V4 PUTs
		if(onUpdate || (onInsert && !CqnService.EVENT_UPSERT.equals(event))) {
			return true;
		}

		// Check FieldControl ReadOnly annotations
		boolean readOnly = CdsAnnotations.READONLY.isTrue(element);
		boolean fieldControlReadOnly = CdsAnnotations.FIELD_CONTROL_READONLY.isTrue(element);
		Map<?, ?> commonFieldControl = CdsAnnotations.COMMON_FIELDCONTROL.getOrDefault(element);
		if(readOnly || fieldControlReadOnly || (commonFieldControl != null && "ReadOnly".equals(commonFieldControl.get("#")))) {
			return true;
		}

		// Core.Computed is read only
		// onUpdate and onInsert are also Core.Computed, but already handled above
		boolean coreComputed = CdsAnnotations.CORE_COMPUTED.isTrue(element);
		if(coreComputed && !onUpdate && !onInsert && !element.isKey()) {
			return true;
		}

		// TODO what about UPSERT?
		// Core.Immutable is read only during update
		boolean coreImmutable = CdsAnnotations.CORE_IMMUTABLE.isTrue(element);
		if(coreImmutable && CqnService.EVENT_UPDATE.equals(event) && !(map.containsKey(Drafts.HAS_ACTIVE_ENTITY) && Boolean.FALSE.equals(map.get(Drafts.HAS_ACTIVE_ENTITY)))) {
			return true;
		}

		return false;
	}

}
