/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.model;

import static java.util.stream.Collectors.joining;

import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.ModelId;
import com.sap.cds.mtx.impl.MetaDataAccessorImpl.CdsModelCreator;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsModelReader;
import com.sap.cds.services.impl.utils.CdsModelUtils;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsModelProvider;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.model.DynamicModelUtils;

/**
 * A {@link CdsModelProvider} that uses the Sidecar to retrieve tenant dependent models.
 */
public class DynamicModelProvider implements CdsModelProvider {

	private static final Logger logger = LoggerFactory.getLogger(DynamicModelProvider.class);
	public static final FeatureTogglesInfo STATIC_MODEL_ACCESS_FEATURE = FeatureTogglesInfo.all();


	private final CdsRuntime runtime;
	private final DynamicModelUtils utils;
	private final MetaDataAccessor<String> accessor;

	public DynamicModelProvider(CdsRuntime runtime) {
		this.runtime = runtime;
		this.utils = new DynamicModelUtils(runtime);
		this.accessor = utils.createMetadataAccessor(null, new CdsModelCreator() {

			@Override
			public CdsModel parse(String csn) {
				return CdsModelReader.read(CdsModelUtils.buildCdsModelReaderConfig(runtime), csn, true);
			}

			@Override
			public CdsModel getBaseModel() {
				// TODO this is not the base model as interpreted by MTXS, as it contains all feature toggles
				return runtime.getCdsModel();
			}

		}, null);
	}

	@Override
	public CdsModel get(UserInfo userInfo, FeatureTogglesInfo featureTogglesInfo) {
		if(featureTogglesInfo == STATIC_MODEL_ACCESS_FEATURE || utils.useStaticModel(userInfo, featureTogglesInfo)) {
			return runtime.getCdsModel();
		} else {
			String tenant = userInfo.getTenant();
			if (logger.isDebugEnabled()) {
				String featureToggles = featureTogglesInfo.getFeatureToggles().map(ft -> ft.getName() + ": " + ft.isEnabled()).collect(joining(","));
				logger.debug("Fetching the model for tenant '{}' and feature toggles '{}'", tenant, featureToggles);
			}
			Set<String> features = featureTogglesInfo.getEnabledFeatureToggles().map(ft -> ft.getName()).collect(Collectors.toSet());
			ModelId modelId = ModelId.create(tenant).features(features).build();
			int maxAgeSeconds = Integer.MAX_VALUE;
			return accessor.getCdsModel(modelId, maxAgeSeconds);
		}
	}
}
