/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.outbox;

import static com.sap.cds.services.utils.CdsErrorStatuses.ASYNC_SERVICE_INTERFACE_DECLARES_UNKNOWN_METHOD;
import static com.sap.cds.services.utils.CdsErrorStatuses.OUTBOX_SERVICE_NOT_OUTBOXABLE;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.sap.cds.services.Service;
import com.sap.cds.services.outbox.OutboxService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

public class OutboxedServiceProxyUtils {

	private OutboxedServiceProxyUtils() {
		// prevent instantiation
	}

	@SuppressWarnings("unchecked")
	public static <S extends Service> S unboxed(S service) {
		if (Proxy.isProxyClass(service.getClass())) {
			InvocationHandler invocationHandler = Proxy.getInvocationHandler(service);
			if (invocationHandler instanceof OutboxedServiceInvocationHandler serviceInvocationHandler) {
				return (S) serviceInvocationHandler.getDelegatedService();
			}
		}
		return service;
	}

	@SuppressWarnings("unchecked")
	public static <S extends Service> S outboxed(OutboxService outboxService, S service, CdsRuntime runtime) {
		ensureNotOutboxService(service);

		S serviceForOutboxing = service;
		if (Proxy.isProxyClass(service.getClass())) {
			InvocationHandler invocationHandler = Proxy.getInvocationHandler(service);
			if (invocationHandler instanceof OutboxedServiceInvocationHandler serviceInvocationHandler) {
				if (serviceInvocationHandler.getOutboxService() == outboxService) {
					return service; // already outboxed with correct outbox
				}
				serviceForOutboxing = (S) serviceInvocationHandler.getDelegatedService();
			}
		}

		OutboxedServiceInvocationHandler invocationHandler = new OutboxedServiceInvocationHandler(outboxService, serviceForOutboxing, runtime);
		return (S) Proxy.newProxyInstance(service.getClass().getClassLoader(), getInterfaces(service), invocationHandler);
	}

	@SuppressWarnings("unchecked")
	public static <A extends Service> A outboxed(OutboxService outboxService, Service service, Class<A> asyncInterface, CdsRuntime runtime) {
		ensureNotOutboxService(service);

		// if interface matches given service delegate to the other method
		if (asyncInterface.isAssignableFrom(service.getClass())) {
			return (A) outboxed(outboxService, service, runtime);
		}

		// no fast path in case of same outbox, as asyncInterface is not implemented. Therefore we need to wrap again anyways.
		Service serviceForOutboxing = unboxed(service);
		Map<Method, Method> methodMapping = getAsyncInterfaceMethodMapping(service, asyncInterface);
		OutboxedAsyncServiceInvocationHandler invocationHandler = new OutboxedAsyncServiceInvocationHandler(outboxService, serviceForOutboxing, methodMapping, runtime);
		return (A) Proxy.newProxyInstance(service.getClass().getClassLoader(), new Class<?>[] { asyncInterface }, invocationHandler);
	}

	private static <S extends Service> void ensureNotOutboxService(S service) {
		if (service instanceof OutboxService) {
			throw new ErrorStatusException(OUTBOX_SERVICE_NOT_OUTBOXABLE, service.getName());
		}
	}

	private static Map<Method, Method> getAsyncInterfaceMethodMapping(Service service, Class<? extends Service> asyncInterface) {
		if (!asyncInterface.isInterface()) {
			throw new ErrorStatusException(CdsErrorStatuses.ASYNC_INTERFACE_NO_INTERFACE, asyncInterface.getName());
		}

		Map<Method, Method> methodMapping = new HashMap<>();
		List<String> missingMethodReferences = new ArrayList<>();
		Class<? extends Service> serviceClass = service.getClass();
		for (Method asyncMethod : asyncInterface.getMethods()) {
			try {
				if (!Modifier.isStatic(asyncMethod.getModifiers())) {
					Method serviceMethod = serviceClass.getMethod(asyncMethod.getName(), asyncMethod.getParameterTypes());
					methodMapping.put(asyncMethod, serviceMethod);
				}
			} catch (NoSuchMethodException e) {
				missingMethodReferences.add(asyncMethod.getName());
			}
		}

		if (!missingMethodReferences.isEmpty()) {
			throw new ErrorStatusException(
					ASYNC_SERVICE_INTERFACE_DECLARES_UNKNOWN_METHOD,
					asyncInterface.getName(),
					String.join(", ", missingMethodReferences),
					service.getName());
		}
		return methodMapping;
	}

	private static Class<?>[] getInterfaces(Service service) {
		Set<Class<?>> interfaces = new LinkedHashSet<>();
		Class<?> clazz = service.getClass();

		do {
			Set<Class<?>> nextInterfaces = new LinkedHashSet<>(Arrays.asList(clazz.getInterfaces()));
			if (interfaces.size() == 1) {
				boolean allAssignable = true;
				Class<?> current = interfaces.iterator().next();
				for (Class<?> nextInterface : nextInterfaces) {
					if (!nextInterface.isAssignableFrom(current)) {
						allAssignable = false;
						break;
					}
				}
				if (!allAssignable) {
					interfaces.addAll(nextInterfaces);
				}
			} else {
				interfaces.addAll(nextInterfaces);
			}
			clazz = clazz.getSuperclass();
		} while (clazz != null);

		return interfaces.toArray(new Class<?>[0]);
	}
}
