/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.runtime;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.impl.changeset.ChangeSetContextImpl;
import com.sap.cds.services.utils.OpenTelemetryUtils;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.ChangeSetContextRunner;

import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Scope;

public class ChangeSetContextRunnerImpl implements ChangeSetContextRunner {

	private final static Logger logger = LoggerFactory.getLogger(CdsRuntimeImpl.class);

	private final CdsRuntime runtime;

	public ChangeSetContextRunnerImpl(CdsRuntime runtime) {
		this.runtime = Objects.requireNonNull(runtime, "runtime must not be null");
	}

	@Override
	public <T> T run(Function<ChangeSetContext, T> changeSetHandler) {
		if(!RequestContext.isActive()) {
			return runtime.requestContext().run(requestContext -> {
				return open(changeSetHandler);
			});
		}
		return open(changeSetHandler);

	}

	@Override
	public void run(Consumer<ChangeSetContext> changeSetHandler) {
		run((changeSetContext) -> {
			changeSetHandler.accept(changeSetContext);
			return Void.TYPE;
		});
	}

	private <T> T open(Function<ChangeSetContext, T> handler) {
		Optional<Span> span = OpenTelemetryUtils.createSpan(OpenTelemetryUtils.CdsSpanType.CHANGESET_CONTEXT);

		//scope needs to be first so that it gets closed after changeset
		try (Scope scope = span.map(Span::makeCurrent).orElse(null);
				ChangeSetContextImpl changeSetContext = ChangeSetContextImpl.open()) {

			OpenTelemetryUtils.updateSpan(span, changeSetContext);
			try {
				return handler.apply(changeSetContext);
			} catch (Exception e) { // NOSONAR
				// cancel in case of any exception / reject
				changeSetContext.markForCancel();
				logger.info("Exception marked the ChangeSet {} as cancelled: {}", changeSetContext.getId(), e.getMessage());
				throw e;
			}
		} catch (Exception e) {
			// includes exceptions triggered by close()
			OpenTelemetryUtils.recordException(span, e);
			throw e;
		} finally {
			OpenTelemetryUtils.endSpan(span);
		}
	}

}
