/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cds.services.impl.persistence;

import com.sap.cds.impl.parser.StructDataParser;
import com.sap.cds.ql.Insert;
import com.sap.cds.ql.Upsert;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.ErrorStatus;
import com.sap.cds.services.ServiceException;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CdsTypeUtils;
import com.sap.cds.util.DataUtils;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.io.input.BOMInputStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CsvDataLoader {
    private static final Logger logger = LoggerFactory.getLogger(CsvDataLoader.class);
    private final PersistenceService db;
    private final CdsRuntime runtime;
    private final CdsProperties.DataSource.Csv config;
    private final char[] supportedDelimiters = new char[]{';', ',', '\t'};

    public CsvDataLoader(PersistenceService db, CdsRuntime runtime) {
        this.db = db;
        this.runtime = runtime;
        this.config = runtime.getEnvironment().getCdsProperties().getDataSource().getCsv();
    }

    public void load() {
        if (this.config.isSingleChangeset()) {
            this.runtime.changeSetContext().run(t -> this.load((ChangeSetContext)t));
        } else {
            this.load(null);
        }
    }

    private void load(ChangeSetContext ctx) {
        for (String path : this.config.getPaths()) {
            File folder;
            boolean recursive = false;
            if (path.endsWith("**") && path.length() > 2) {
                path = path.substring(0, path.length() - 2);
                recursive = true;
            }
            if (!(folder = new File(path)).exists() || !folder.isDirectory()) continue;
            this.loadFolder(folder, recursive, ctx);
        }
    }

    public void loadFolder(File folder, boolean recursive, ChangeSetContext ctx) {
        String fileSuffix = this.config.getFileSuffix();
        for (File file : folder.listFiles()) {
            if (file.isFile() && file.getName().endsWith(fileSuffix)) {
                this.loadFile(file, ctx);
            }
            if (!recursive || !file.isDirectory()) continue;
            this.loadFolder(file, true, ctx);
        }
    }

    void loadFile(File file, ChangeSetContext ctx) {
        if (file.length() == 0L) {
            return;
        }
        try (CSV csv = new CSV(file);){
            List<Map<String, Object>> entries = csv.data().collect(Collectors.toList());
            if (!entries.isEmpty()) {
                if (ctx == null) {
                    try {
                        this.runtime.changeSetContext().run(context -> this.loadFile(csv, entries, (ChangeSetContext)context));
                    }
                    catch (ServiceException e) {
                        logger.debug("Skipped filling {} from {}", new Object[]{csv.entity.getQualifiedName(), file.getPath(), e});
                    }
                } else {
                    this.loadFile(csv, entries, ctx);
                }
            }
        }
        catch (IOException | UncheckedIOException e) {
            throw new ErrorStatusException((ErrorStatus)CdsErrorStatuses.INVALID_CSV_FILE, new Object[]{file.getPath(), e});
        }
    }

    private void loadFile(CSV csv, List<Map<String, Object>> entries, ChangeSetContext ctx) {
        if ("always".equals(this.config.getInitializationMode())) {
            Upsert upsert = (Upsert)Upsert.into((CdsEntity)csv.entity()).entries(entries).hint("cross-tenant", (Object)true);
            this.db.run((CqnUpsert)upsert);
        } else {
            Insert insert = (Insert)Insert.into((CdsEntity)csv.entity()).entries(entries).hint("cross-tenant", (Object)true);
            this.db.run((CqnInsert)insert);
        }
        if (ctx.isMarkedForCancel()) {
            logger.debug("Cancelled filling {} from {}", (Object)csv.entity().getQualifiedName(), (Object)csv.file.getPath());
        } else {
            logger.info("Filling {} from {}", (Object)csv.entity().getQualifiedName(), (Object)csv.file.getPath());
        }
    }

    private class CSV
    implements AutoCloseable {
        private final CdsEntity entity;
        private final File file;
        private final BufferedReader br;
        private final List<ElementPath> headers;
        private CSVParser csvParser;
        private final CSVFormat.Builder formatTemplate = CSVFormat.Builder.create((CSVFormat)CSVFormat.RFC4180).setEscape('\\').setCommentMarker('#').setIgnoreEmptyLines(true).setAllowMissingColumnNames(true);

        public CSV(File file) throws IOException {
            this.file = file;
            this.br = new BufferedReader(new InputStreamReader((InputStream)new BOMInputStream(Files.newInputStream(file.toPath(), new OpenOption[0])), StandardCharsets.UTF_8));
            String entityName = this.entityName(file.getName());
            this.entity = CsvDataLoader.this.runtime.getCdsModel().findEntity(entityName).orElseGet(() -> {
                Optional entity;
                String theEntityName = entityName;
                if (theEntityName.lastIndexOf(".texts") > 0 && (entity = CsvDataLoader.this.runtime.getCdsModel().findEntity(theEntityName.substring(0, theEntityName.lastIndexOf(".texts") + 6))).isPresent()) {
                    return (CdsEntity)entity.get();
                }
                if (theEntityName.lastIndexOf("_texts") > 0) {
                    theEntityName = theEntityName.substring(0, theEntityName.lastIndexOf("_texts") + 6);
                    entity = CsvDataLoader.this.runtime.getCdsModel().findEntity(theEntityName);
                    if (entity.isPresent()) {
                        return (CdsEntity)entity.get();
                    }
                }
                if (theEntityName.endsWith("_texts")) {
                    String altEntityName = theEntityName.substring(0, theEntityName.length() - "_texts".length()) + ".texts";
                    return CsvDataLoader.this.runtime.getCdsModel().findEntity(altEntityName).orElse(null);
                }
                return null;
            });
            if (this.entity == null) {
                throw new ErrorStatusException((ErrorStatus)CdsErrorStatuses.INVALID_CSV_FILE_ENTITYNOTFOUND, new Object[]{entityName, file.getPath()});
            }
            this.headers = this.parseHeaders();
        }

        public CdsEntity entity() {
            return this.entity;
        }

        public Stream<Map<String, Object>> data() {
            if (this.headers.isEmpty()) {
                return Stream.empty();
            }
            try {
                return this.csvParser.getRecords().stream().map(this::convert);
            }
            catch (UncheckedIOException e) {
                throw new ErrorStatusException((ErrorStatus)CdsErrorStatuses.INVALID_CSV_FILE, new Object[]{this.file.getPath(), e});
            }
        }

        private List<ElementPath> parseHeaders() throws IOException {
            String headerLine;
            while ((headerLine = this.br.readLine()) != null && headerLine.isEmpty()) {
            }
            if (headerLine == null) {
                return Collections.emptyList();
            }
            String delimiter = this.detectDelimiter(headerLine);
            this.csvParser = new CSVParser((Reader)this.br, this.formatTemplate.setHeader(CSV.splitHeaders(headerLine, delimiter)).setDelimiter(delimiter).build());
            return this.csvParser.getHeaderNames().stream().filter(StringUtils::isNotBlank).map(n -> this.path((CdsStructuredType)this.entity, (String)n)).collect(Collectors.toList());
        }

        private String detectDelimiter(String line) {
            char result = ';';
            int delimitersCount = 0;
            for (char current : CsvDataLoader.this.supportedDelimiters) {
                if (line.indexOf(current) < 0) continue;
                result = current;
                ++delimitersCount;
            }
            if (delimitersCount > 1) {
                throw new ErrorStatusException((ErrorStatus)CdsErrorStatuses.INVALID_CSV_FILE_INVALIDHEADER, new Object[]{this.file.getPath()});
            }
            return String.valueOf(result);
        }

        private Map<String, Object> convert(CSVRecord csvRecord) {
            HashMap<String, Object> row = new HashMap<String, Object>();
            for (ElementPath header : this.headers) {
                String value;
                if (!csvRecord.isSet(header.csvName) || !StringUtils.isNotEmpty((CharSequence)(value = csvRecord.get(header.csvName)))) continue;
                this.handleValue(row, header.element, header.path, value);
            }
            return row;
        }

        private void handleValue(Map<String, Object> row, CdsElement element, String path, String value) {
            CdsType type = element.getType();
            if (type.isSimpleType(CdsBaseType.UUID) && !CdsTypeUtils.isStrictUUID((CdsElement)element, (CdsType)type)) {
                this.direct(row, path, (CdsSimpleType)type.as(CdsSimpleType.class), value);
            } else if (type.isSimple()) {
                this.simple(row, path, (CdsSimpleType)type.as(CdsSimpleType.class), value);
            } else if (type.isStructured()) {
                this.structured(row, path, (CdsStructuredType)type.as(CdsStructuredType.class), value);
            } else if (type.isArrayed()) {
                this.arrayed(row, path, element, value);
            } else if (type.isAssociation()) {
                this.association(row, path, element, value);
            }
        }

        private void direct(Map<String, Object> row, String path, CdsSimpleType type, String value) {
            try {
                DataUtils.resolvePathAndAdd(row, (String)path, (Object)value);
            }
            catch (Exception e) {
                throw this.error(path, (CdsType)type, false, value, e);
            }
        }

        private void simple(Map<String, Object> row, String path, CdsSimpleType type, String value) {
            try {
                Object val = CdsTypeUtils.parse((CdsBaseType)type.getType(), (String)value);
                DataUtils.resolvePathAndAdd(row, (String)path, (Object)val);
            }
            catch (Exception e) {
                throw this.error(path, (CdsType)type, false, value, e);
            }
        }

        private void structured(Map<String, Object> row, String path, CdsStructuredType type, String value) {
            try {
                Map val = StructDataParser.create((CdsStructuredType)type).parseObject(value);
                DataUtils.resolvePathAndAdd(row, (String)path, (Object)val);
            }
            catch (Exception e) {
                throw this.error(path, (CdsType)type, false, value, e);
            }
        }

        private void arrayed(Map<String, Object> row, String path, CdsElement element, String value) {
            CdsType itemsType = ((CdsArrayedType)element.getType().as(CdsArrayedType.class)).getItemsType();
            try {
                List val = StructDataParser.parseArrayOf((CdsType)itemsType, (String)value);
                DataUtils.resolvePathAndAdd(row, (String)path, (Object)val);
            }
            catch (Exception e) {
                throw this.error(path, itemsType, true, value, e);
            }
        }

        private void association(Map<String, Object> row, String path, CdsElement element, String value) {
            CdsEntity targetType = ((CdsAssociationType)element.getType().as(CdsAssociationType.class)).getTarget();
            StructDataParser parser = StructDataParser.create((CdsStructuredType)targetType);
            boolean isSingleValued = CdsModelUtils.isSingleValued((CdsType)element.getType());
            try {
                Object val = isSingleValued ? parser.parseObject(value) : parser.parseArray(value);
                DataUtils.resolvePathAndAdd(row, (String)path, (Object)val);
            }
            catch (Exception e) {
                throw this.error(path, (CdsType)targetType, !isSingleValued, value, e);
            }
        }

        private ErrorStatusException error(String element, CdsType type, boolean array, String value, Exception e) {
            return new ErrorStatusException((ErrorStatus)CdsErrorStatuses.INVALID_CSV_FILE_TYPEMISMATCH, new Object[]{value, array ? type.getName() + "[]" : type.getName(), element, this.file.getPath(), e});
        }

        private String entityName(String fileName) {
            return fileName.replace("-", ".").substring(0, fileName.length() - 4);
        }

        private ElementPath path(CdsStructuredType entity, String column) {
            String elementName = column.trim();
            Optional<ElementPath> path = this.findElement(entity, elementName).map(p -> new ElementPath(entity.getElement(p), (String)p, column));
            return path.orElseThrow(() -> new ErrorStatusException((ErrorStatus)CdsErrorStatuses.INVALID_CSV_FILE_UNKNOWNCOLUMN, new Object[]{this.file.getPath(), elementName, entity.getQualifiedName()}));
        }

        private Optional<String> findElement(CdsStructuredType entity, String column) {
            column = column.toUpperCase(Locale.US);
            Map<String, CdsElement> normalized = this.normalize(entity);
            CdsElement element = normalized.get(column);
            if (element != null) {
                return Optional.of(element.getName());
            }
            for (Map.Entry<String, CdsElement> entry : normalized.entrySet()) {
                String n = entry.getKey();
                CdsElement e = entry.getValue();
                if (!column.startsWith(n + "_")) continue;
                String suffix = column.substring(n.length() + 1);
                CdsType type = e.getType();
                Optional<Object> subElement = Optional.empty();
                if (type.isStructured()) {
                    subElement = this.findElement((CdsStructuredType)type.as(CdsStructuredType.class), suffix);
                } else if (CdsModelUtils.managedToOne((CdsType)type)) {
                    subElement = this.findElement((CdsStructuredType)((CdsAssociationType)type.as(CdsAssociationType.class)).getTarget(), suffix);
                }
                if (!subElement.isPresent()) continue;
                String path = e.getName() + "." + (String)subElement.get();
                return Optional.of(path);
            }
            return Optional.empty();
        }

        private Map<String, CdsElement> normalize(CdsStructuredType type) {
            return type.elements().collect(Collectors.toMap(e -> e.getName().toUpperCase(Locale.US), e -> e));
        }

        @Override
        public void close() throws IOException {
            if (this.csvParser != null) {
                this.csvParser.close();
            }
            if (this.br != null) {
                this.br.close();
            }
        }

        private static String[] splitHeaders(String headerLine, String delimiter) {
            return (String[])Stream.of(headerLine.split(delimiter)).map(v -> StringUtils.unwrap((String)v, (char)CSVFormat.RFC4180.getQuoteCharacter().charValue())).toArray(String[]::new);
        }
    }

    private static class ElementPath {
        final CdsElement element;
        final String path;
        final String csvName;

        public ElementPath(CdsElement element, String path, String csvName) {
            this.element = element;
            this.path = path;
            this.csvName = csvName;
        }
    }
}

